<?php

namespace Wi\Admin\AdminBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Http\Event\InteractiveLoginEvent;
use Wi\Admin\AdminBundle\Entity\Administrator;
use Wi\Admin\AdminBundle\Entity\Permission;
use Wi\Admin\AdminBundle\Form\AdministratorType;
use Wi\Admin\AdminBundle\Form\AdministratorDeleteType;
use Wi\Admin\AdminBundle\Form\PermissionType;

/**
 * Administrator controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class AdministratorController extends Controller
{
    /**
     * Lista administratorów.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction()
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich administratorów.
        $administrators = $em->getRepository('WiAdminAdminBundle:Administrator')
            ->findAllActive()
        ;

        return $this->render('WiAdminAdminBundle:Administrator:index.html.php', [
            'administrators' => $administrators,
        ]);
    }

    /**
     * Dodawanie nowego administratora.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Wyłączenie filtra softdeleteable.
        $em->getFilters()->disable('softdeleteable');

        // Utworzenie nowego obiektu administratora.
        $administrator = new Administrator();

        // Generowanie formularza.
        $form = $this->createForm(AdministratorType::class, $administrator);
        $form->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Ustawienie hasła administratora.
            $administrator->setPassword(
                $this->get('security.password_encoder')
                    ->encodePassword(
                        $administrator,
                        $administrator->getPassword()
                    )
            );

            // Zapis.
            $em->persist($administrator);
            $em->flush();

            return $this->redirectToRoute('wi_admin_admin_administrators_show', [
                'id' => $administrator->getId()
            ]);
        }

        // Włączenie filtra softdeleteable.
        $em->getFilters()->enable('softdeleteable');

        return $this->render('WiAdminAdminBundle:Administrator:new.html.php', [
            'addNewForm' => $form->createView(),
        ]);
    }

    /**
     * Edytowanie administratora.
     *
     * @param   Request         $request
     * @param   Administrator   $administrator
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Administrator $administrator)
    {
        // Sprawdzenie, czy administrator jest aktywny.
        if ($administrator->getStatus() != 'A') {
            throw $this->createNotFoundException('Podana strona nie została odnaleziona.');
        }

        // Tymczasowe przechowanie hasła.
        $password = $administrator->getPassword();

        // Generowanie formularza.
        $editForm = $this->createForm(AdministratorType::class, $administrator);
        $editForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($editForm->isSubmitted() && $editForm->isValid()) {
            if (! is_null($editForm->get('password')->getData())) {
                // Ustawienie hasła administratora.
                $administrator->setPassword(
                    $this->get('security.password_encoder')
                        ->encodePassword(
                            $administrator,
                            $administrator->getPassword()
                        )
                );
            } else {
                $administrator->setPassword($password);
            }

            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($administrator);
            $em->flush();

            return $this->redirectToRoute('wi_admin_admin_administrators_show', [
                'id' => $administrator->getId()
            ]);
        }

        return $this->render('WiAdminAdminBundle:Administrator:edit.html.php', [
            'addEditForm' => $editForm->createView(),
            'administrator' => $administrator,
        ]);
    }

    /**
     * Edytowanie uprawnień administratora.
     *
     * @param   Request         $request
     * @param   Administrator   $administrator
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function editPermissionAction(Request $request, Administrator $administrator)
    {
        // Generowanie formularza.
        $formPermission = $this->createForm(PermissionType::class, $administrator);
        $formPermission->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($formPermission->isSubmitted() && $formPermission->isValid()) {
            $roles = $formPermission->get('roles')->getData();

            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Sprawdzenie, czy przesłane opcje nie są puste.
            if (! empty($roles)) {
                foreach ($roles as $role) {
                    // Sprawdzenie, czy administrator posiada dane uprawnienie.
                    $hasRole = $administrator->getPermissions()->filter(
                        function ($entry) use ($role) {
                            return $entry->getRole() === $role;
                        }
                    );

                    // Jeżeli administrator jeszcze nie posiada takiego uprawnienia.
                    if ($hasRole->count() == 0) {
                        // Utworzenie nowego obiektu uprawnień.
                        $permission = new Permission();

                        // Przypisanie roli do obiektu uprawnień.
                        $permission->setRole($role);

                        // Przypisanie administratora do obiektu uprawnień.
                        $permission->setAdministrator($administrator);

                        // Przygotowanie do zapisu.
                        $em->persist($permission);
                    }
                }
            }

            foreach ($administrator->getPermissions() as $permission) {
                $hasPermission = $roles->contains($permission->getRole());

                // Sprawdzenie, czy administratorowi usunięto uprawnienia.
                if (! $hasPermission) {
                    // Przygotowanie do usunięcia.
                    $em->remove($permission);
                }
            }

            // Przygotowanie do zapisu.
            $em->persist($administrator);

            // Zapis.
            $em->flush();

            return $this->redirectToRoute('wi_admin_admin_administrators_show', [
                'id' => $administrator->getId()
            ]);
        }

        return $this->render('WiAdminAdminBundle:Administrator:edit_permission.html.php', [
            'administrator' => $administrator,
            'formPermission' => $formPermission->createView(),
        ]);
    }

    /**
     * Usuwanie administratora.
     *
     * @param   Request         $request
     * @param   Administrator   $administrator
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, Administrator $administrator)
    {
        // Sprawdzenie, czy administrator jest aktywny lub jest aktualnie zalogowany.
        if (
            $administrator->getStatus() != 'A' ||
            $this->getUser()->getId() == $administrator->getId()
        ) {
            throw $this->createNotFoundException('Podana strona nie została odnaleziona.');
        }

        // Generowanie formularza.
        $deleteForm = $this->createForm(AdministratorDeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->remove($administrator);
            $em->flush();

            return $this->redirectToRoute('wi_admin_admin_administrators_index');
        }

        return $this->render('WiAdminAdminBundle:Administrator:delete.html.php', [
            'administrator' => $administrator,
            'formDelete' => $deleteForm->createView(),
        ]);
    }

    /**
     * Wyświetla szeczegóły administratora.
     *
     * @param   Administrator $administrator
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Administrator $administrator)
    {
        // Sprawdzenie, czy administrator jest aktywny.
        if ($administrator->getStatus() != 'A') {
            throw $this->createNotFoundException('Podana strona nie została odnaleziona.');
        }

        return $this->render('WiAdminAdminBundle:Administrator:show.html.php', [
            'administrator' => $administrator,
        ]);
    }

    /**
     * Zmiana hasła administratora z okna modalnego.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function resetPasswordAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie danych z formularza.
        $password = $request->get('password');
        $rePassword = $request->get('rePassword');

        $admin = $this->getUser();
        $save = false;
        // Sprawdzenie czy nowe hasło nie jest puste lub podane hasła nie są różne.
        if(! is_null($password) && $password == $rePassword){

            $admin->setPassword(
                $this->get('security.password_encoder')
                    ->encodePassword(
                        $admin,
                        $password
                    )
            );

            // Zapis.
            $em->persist($admin);
            $em->flush();
            $save = true;
        }


        return new JsonResponse($save);
    }

    /**
     * Zmiana języka z okna modalnego.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function changeLanguageAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie danych z formularza.
        $locale = $request->get('language');

        $admin = $this->getUser();
        $save =  false;
        if(! is_null($locale)){
            $admin->setLocale($locale);
            // Zapis.
            $em->persist($admin);
            $em->flush();
            $save = true;
            $request->getSession()->set('_locale', $locale);
        }

        return new JsonResponse($save);
    }
}
