<?php

namespace Wi\Admin\AdminBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\ArrayCollection;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Validator\Constraints as Assert;
use Wi\Admin\AdminBundle\Entity\Permission;

/**
 * Administrator
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 *
 * @ORM\Table(
 *      name="admin_administrators",
 *      uniqueConstraints={
 *          @ORM\UniqueConstraint(name="UQ_Administrators_Login",columns={"login"}),
 *          @ORM\UniqueConstraint(name="UQ_Administrators_Email",columns={"email"}),
 *          @ORM\UniqueConstraint(name="UQ_Administrators_Hash",columns={"hash"})
 *      }
 * )
 * @ORM\Entity(repositoryClass="Wi\Admin\AdminBundle\Repository\AdministratorRepository")
 * @ORM\EntityListeners({"Wi\Admin\AdminBundle\EventListener\Entity\AdministratorListener"})
 * @ORM\HasLifecycleCallbacks()
 * @UniqueEntity(fields="login", message="adminEntity.errors.login")
 * @UniqueEntity(fields="email", message="adminEntity.errors.email")
 * @Gedmo\SoftDeleteable(fieldName = "dateDeleted")
 */
class Administrator implements UserInterface, \Serializable
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="login", type="string", length=64)
     * @Assert\NotBlank()
     */
    private $login;

    /**
     * @var string
     *
     * @ORM\Column(name="password", type="string", length=256)
     */
    private $password;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateCreated", type="datetime")
     */
    private $dateCreated;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateDeleted", type="datetime", nullable=true)
     */
    private $dateDeleted;

    /**
     * @var string
     *
     * @ORM\Column(name="status", type="string", length=1, options={"default": "A", "fixed" = true})
     */
    private $status;

    /**
     * @var string
     *
     * @ORM\Column(name="firstName", type="string", length=45, nullable=true)
     */
    private $firstName;

    /**
     * @var string
     *
     * @ORM\Column(name="lastName", type="string", length=45, nullable=true)
     */
    private $lastName;

    /**
     * @var string
     *
     * @ORM\Column(name="displayName", type="string", length=125, nullable=true)
     */
    private $displayName;

    /**
     * @var string
     *
     * @ORM\Column(name="email", type="string", length=64)
     * @Assert\Email()
     * @Assert\NotBlank()
     */
    private $email;

    /**
     * @var string
     *
     * @ORM\Column(name="job", type="string", length=128, nullable=true)
     */
    private $job;

    /**
     * @var string
     *
     * @ORM\Column(name="hash", type="string", length=64, nullable=true)
     */
    private $hash;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="hashExpired", type="datetime", nullable=true)
     */
    private $hashExpired;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\AdminBundle\Entity\Permission", mappedBy="administrator", fetch="LAZY")
     */
    private $permissions;

    /**
     * @var string
     *
     * @ORM\Column(name="locale", type="string", length=2)
     */
    private $locale;


    // ------------------------------------------------------------
    // Has Lifecycle Callbacks methods.
    // ------------------------------------------------------------

    /**
     * Funkcja wykonuje się podczas wstawiania nowego rekordu do bazy.
     *
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        // Ustawienie daty utworzenia.
        $this->dateCreated = new \DateTime();

        // Ustawienie statusu.
        if (is_null($this->status)) {
            $this->status = 'A';
        }

        // Ustawienie języka.
        if (is_null($this->locale)) {
            $this->locale = 'en';
        }
    }

    // ------------------------------------------------------------
    // User Interface methods.
    // ------------------------------------------------------------

    public function getSalt()
    {
        // you *may* need a real salt depending on your encoder
        // see section on salt below
        return null;
    }

    public function eraseCredentials()
    {
    }

    /** @see \Serializable::serialize() */
    public function serialize()
    {
        return serialize(array(
            $this->id,
            $this->login,
            $this->password,
            // see section on salt below
            // $this->salt,
        ));
    }

    /** @see \Serializable::unserialize() */
    public function unserialize($serialized)
    {
        list (
            $this->id,
            $this->login,
            $this->password,
            // see section on salt below
            // $this->salt
        ) = unserialize($serialized);
    }

    public function getRoles()
    {
        $arr = ['ROLE_ADMIN'];

        foreach ($this->getPermissions() as $permission) {
            $arr[] = $permission->getRole()->getType();
        }

        return $arr;
    }

    public function getUsername()
    {
        return $this->login;
    }

    // ------------------------------------------------------------
    // Getters and setters methods.
    // ------------------------------------------------------------

    /**
     * Set id
     *
     * @param int $id
     * @return Administrator
     */
    public function setId($id)
    {
        $this->id = $id;

        return $this;
    }

    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set login
     *
     * @param string $login
     * @return Administrator
     */
    public function setLogin($login)
    {
        $this->login = $login;

        return $this;
    }

    /**
     * Get login
     *
     * @return string
     */
    public function getLogin()
    {
        return $this->login;
    }

    /**
     * Set password
     *
     * @param string $password
     * @return Administrator
     */
    public function setPassword($password)
    {
        $this->password = $password;

        return $this;
    }

    /**
     * Get password
     *
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * Set dateCreated
     *
     * @param \DateTime $dateCreated
     * @return Administrator
     */
    public function setDateCreated($dateCreated)
    {
        $this->dateCreated = $dateCreated;

        return $this;
    }

    /**
     * Get dateCreated
     *
     * @return \DateTime
     */
    public function getDateCreated()
    {
        return $this->dateCreated;
    }

    /**
     * Set dateDeleted
     *
     * @param \DateTime $dateDeleted
     * @return Administrator
     */
    public function setDateDeleted($dateDeleted)
    {
        $this->dateDeleted = $dateDeleted;

        return $this;
    }

    /**
     * Get dateDeleted
     *
     * @return \DateTime
     */
    public function getDateDeleted()
    {
        return $this->dateDeleted;
    }

    /**
     * Set status
     *
     * @param string $status
     * @return Administrator
     */
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set firstName
     *
     * @param string $firstName
     * @return Administrator
     */
    public function setFirstName($firstName)
    {
        $this->firstName = $firstName;

        return $this;
    }

    /**
     * Get firstName
     *
     * @return string
     */
    public function getFirstName()
    {
        return $this->firstName;
    }

    /**
     * Set lastName
     *
     * @param string $lastName
     * @return Administrator
     */
    public function setLastName($lastName)
    {
        $this->lastName = $lastName;

        return $this;
    }

    /**
     * Get lastName
     *
     * @return string
     */
    public function getLastName()
    {
        return $this->lastName;
    }

    /**
     * Set displayName
     *
     * @param string $displayName
     * @return Administrator
     */
    public function setDisplayName($displayName)
    {
        $this->displayName = $displayName;

        return $this;
    }

    /**
     * Get displayName
     *
     * @return string
     */
    public function getDisplayName()
    {
        return $this->displayName;
    }

    /**
     * Set email
     *
     * @param string $email
     * @return Administrator
     */
    public function setEmail($email)
    {
        $this->email = $email;

        return $this;
    }

    /**
     * Get email
     *
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * Set job
     *
     * @param string $job
     * @return Administrator
     */
    public function setJob($job)
    {
        $this->job = $job;

        return $this;
    }

    /**
     * Get job
     *
     * @return string
     */
    public function getJob()
    {
        return $this->job;
    }

    /**
     * Set hash
     *
     * @param string $hash
     * @return Administrator
     */
    public function setHash($hash)
    {
        $this->hash = $hash;

        return $this;
    }

    /**
     * Get hash
     *
     * @return string
     */
    public function getHash()
    {
        return $this->hash;
    }

    /**
     * Set hashExpired
     *
     * @param \DateTime $hashExpired
     * @return Administrator
     */
    public function setHashExpired($hashExpired)
    {
        $this->hashExpired = $hashExpired;

        return $this;
    }

    /**
     * Get hashExpired
     *
     * @return \DateTime
     */
    public function getHashExpired()
    {
        return $this->hashExpired;
    }

    /**
     * Add permission
     *
     * @param Permission $permission
     * @return Administrator
     */
    public function addPermission(Permission $permission)
    {
        $this->permissions[] = $permission;

        return $this;
    }

    /**
     * Remove permission
     *
     * @param Permission $permission
     */
    public function removePermission(Permission $permission)
    {
        $this->permissions->removeElement($permission);
    }

    /**
     * Get permissions
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getPermissions()
    {
        return $this->permissions;
    }

    /**
     * Set locale
     *
     * @param string $locale
     * @return Administrator
     */
    public function setLocale($locale)
    {
        $this->locale = $locale;

        return $this;
    }

    /**
     * Get locale
     *
     * @return string
     */
    public function getLocale()
    {
        return $this->locale;
    }

    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->permissions = new ArrayCollection();
    }

    /**
     * Pobiera nazwę administratora do listy wyboru. Nazwa do wyświetlenia,
     * jeśli nie ma to sprawdza imię, jeśli nie ma to zostaje login.
     * Jeśli jest imię to sprawdza czy jest nazwisko i je dodaje.
     *
     * @return string
     */
    public function getName() : string
    {
        if (! $name = $this->displayName) {
            if (! $name = $this->firstName) {
                $name = $this->login;
            } else {
                $name .= ($this->lastName ? ' ' . $this->lastName : '');
            }
        }

        return $name;
    }
}
