<?php

namespace Wi\Admin\AdvertisingBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Wi\Admin\AdvertisingBundle\Entity\AdvertisingBlock;

/**
 * Advertisement
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 *
 * @ORM\Table(name="advertisements")
 * @ORM\Entity(repositoryClass="Wi\Admin\AdvertisingBundle\Repository\AdvertisementRepository")
 * @ORM\HasLifecycleCallbacks()
 * @Gedmo\SoftDeleteable(fieldName = "dateDeleted")
 */
class Advertisement
{
    /**
     * Status reklamy - aktywna.
     *
     * @const STATUS_ACTIVE
     */
    const STATUS_ACTIVE = 'A';

    /**
     * Status reklamy - usunięta.
     *
     * @const STATUS_DELETED
     */
    const STATUS_DELETED = 'D';

    /**
     * Status reklamy - wygasła.
     *
     * @const STATUS_EXPIRED
     */
    const STATUS_EXPIRED = 'E';

    /**
     * Status reklamy - w trakcie.
     *
     * @const STATUS_PENDING
     */
    const STATUS_PENDING = 'P';

    // -------------------------------------------------------------------------

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var AdvertisingBlock
     *
     * @ORM\ManyToOne(targetEntity="Wi\Admin\AdvertisingBundle\Entity\AdvertisingBlock", inversedBy="advertisements")
     * @ORM\JoinColumn(name="advertisingBlock_id", referencedColumnName="id", nullable=false)
     */
    private $advertisingBlock;

    /**
     * @var string
     *
     * @ORM\Column(name="html", type="text", nullable=true)
     */
    private $html;

    /**
     * @var string
     *
     * @ORM\Column(name="image", type="string", length=255, nullable=true)
     */
    private $image;

    /**
     * @var string
     *
     * @ORM\Column(name="link", type="string", length=511, nullable=true)
     */
    private $link;

    /**
     * @var int
     *
     * @ORM\Column(name="orderNr", type="integer", nullable=true)
     */
    private $orderNr;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateStart", type="datetime", nullable=true)
     */
    private $dateStart;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateEnd", type="datetime", nullable=true)
     */
    private $dateEnd;

    /**
     * @var string
     *
     * @ORM\Column(name="status", type="string", length=1, options={"fixed" = true})
     */
    private $status;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateCreated", type="datetime")
     */
    private $dateCreated;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateDeleted", type="datetime", nullable=true)
     */
    private $dateDeleted;


    // ------------------------------------------------------------
    // Has Lifecycle Callbacks methods.
    // ------------------------------------------------------------

    /**
     * Funkcja wykonuje się podczas wstawiania nowego rekordu do bazy.
     *
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        // Ustawienie daty utworzenia.
        $this->dateCreated  = new \DateTime();

        // Ustawienie domyślnego statusu.
        if (is_null($this->status)) {
            $this->status = self::STATUS_PENDING;
        }

        // Ustawienie pól w zależności od typu bloku.
        switch ($this->advertisingBlock->getType()) {
            case AdvertisingBlock::TYPE_HTML:
                $this->image = null;
                $this->link = null;
                break;
            case AdvertisingBlock::TYPE_BANNER:
                // no break
            case AdvertisingBlock::TYPE_SLIDER:
                $this->html = null;
                break;
        }
    }

    // ------------------------------------------------------------
    // Getters and setters methods.
    // ------------------------------------------------------------

    /**
     * Set id
     *
     * @param int $id
     * @return int
     */
    public function setId($id)
    {
        $this->id = $id;

        return $this;
    }

    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set advertisingBlock
     *
     * @param AdvertisingBlock $advertisingBlock
     * @return Advertisement
     */
    public function setAdvertisingBlock(AdvertisingBlock $advertisingBlock)
    {
        $this->advertisingBlock = $advertisingBlock;

        return $this;
    }

    /**
     * Get advertisingBlock
     *
     * @return AdvertisingBlock
     */
    public function getAdvertisingBlock()
    {
        return $this->advertisingBlock;
    }

    /**
     * Set html
     *
     * @param string $html
     * @return Advertisement
     */
    public function setHtml($html)
    {
        $this->html = $html;

        return $this;
    }

    /**
     * Get html
     *
     * @return string
     */
    public function getHtml()
    {
        return $this->html;
    }

    /**
     * Set image
     *
     * @param string $image
     * @return Advertisement
     */
    public function setImage($image)
    {
        $this->image = $image;

        return $this;
    }

    /**
     * Get image
     *
     * @return string
     */
    public function getImage()
    {
        return $this->image;
    }

    /**
     * Set link
     *
     * @param string $link
     * @return Advertisement
     */
    public function setLink($link)
    {
        $this->link = $link;

        return $this;
    }

    /**
     * Get link
     *
     * @return string
     */
    public function getLink()
    {
        return $this->link;
    }

    /**
     * Set orderNr
     *
     * @param integer $orderNr
     * @return Advertisement
     */
    public function setOrderNr($orderNr)
    {
        $this->orderNr = $orderNr;

        return $this;
    }

    /**
     * Get orderNr
     *
     * @return int
     */
    public function getOrderNr()
    {
        return $this->orderNr;
    }

    /**
     * Set dateStart
     *
     * @param \DateTime $dateStart
     * @return Advertisement
     */
    public function setDateStart($dateStart)
    {
        $this->dateStart = $dateStart;

        return $this;
    }

    /**
     * Get dateStart
     *
     * @return \DateTime
     */
    public function getDateStart()
    {
        return $this->dateStart;
    }

    /**
     * Set dateEnd
     *
     * @param \DateTime $dateEnd
     * @return Advertisement
     */
    public function setDateEnd($dateEnd)
    {
        $this->dateEnd = $dateEnd;

        return $this;
    }

    /**
     * Get dateEnd
     *
     * @return \DateTime
     */
    public function getDateEnd()
    {
        return $this->dateEnd;
    }

    /**
     * Set status
     *
     * @param string $status
     * @return Advertisement
     */
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set dateCreated
     *
     * @param \DateTime $dateCreated
     * @return Advertisement
     */
    public function setDateCreated($dateCreated)
    {
        $this->dateCreated = $dateCreated;

        return $this;
    }

    /**
     * Get dateCreated
     *
     * @return \DateTime
     */
    public function getDateCreated()
    {
        return $this->dateCreated;
    }

    /**
     * Set dateDeleted
     *
     * @param \DateTime $dateDeleted
     * @return Advertisement
     */
    public function setDateDeleted($dateDeleted)
    {
        $this->dateDeleted = $dateDeleted;

        return $this;
    }

    /**
     * Get dateDeleted
     *
     * @return \DateTime
     */
    public function getDateDeleted()
    {
        return $this->dateDeleted;
    }

    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Zwraca przyjazną datę dla użytkownika.
     *
     * @param \DateTime $datetime
     * @return string|null
     */
    public function getDateFormat(\DateTime $datetime = null)
    {
        if (! is_null($datetime)) {
            return $datetime->format('Y-m-d');
        }

        return null;
    }

    /**
     * Zwraca zawartość dla bloku reklamowego w zależności od jego typu.
     *
     * @return string
     */
    public function getContent()
    {
        switch ($this->advertisingBlock->getType()) {
            case AdvertisingBlock::TYPE_HTML: // HTML.
                return $this->html;
            case AdvertisingBlock::TYPE_BANNER: // Baner.
                // no break
            case AdvertisingBlock::TYPE_SLIDER: // Slider.
                return $this->getImageLink();
        }

        return null;
    }

    /**
     * Zwraca ścieżkę relatywną do zdjęcia.
     *
     * @return string
     */
    public function getImageLink()
    {
        return ($this->image ? 'media/assets/Commercial/'.$this->image : null);
    }
}
