<?php

namespace Wi\Admin\BannerBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\BannerBundle\Entity\Banner;
use Wi\Admin\BannerBundle\Form\BannerDeleteType;
use Wi\Admin\BannerBundle\Form\BannerType;

/**
 * Banner controller.
 *
 * @author Maksymilian Baryczka <mmaxioo33@gmail.com>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class BannerController extends Controller
{
    /**
     * Lists all banner entities.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction()
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        $bannerSet = $em->getRepository('WiAdminBannerBundle:BannerSet')->findAll();
        $banners = $em->getRepository('WiAdminBannerBundle:Banner')
            ->findAllActive()
        ;

        return $this->render('WiAdminBannerBundle:Banner:index.html.php', [
            'banners' => $banners,
            'bannerSet' => $bannerSet,
        ]);
    }

    /**
     * Creates a new banner entity.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie nowego obiektu strony.
        $banner = new Banner();

        // Dodanie domyślnego bloku, jeżli została przekazana oraz istnieje.
        if ($request->query->getInt('bannerset_id')) {
            $bannerSet = $em->getRepository('WiAdminBannerBundle:BannerSet')
                ->findOneById($request->query->getInt('bannerset_id'))
            ;

            if (! is_null($bannerSet)) {
                $banner->setBannerSet($bannerSet);
            }
        }

        // Generowanie formularza.
        $form = $this->createForm(BannerType::class, $banner);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Upload file.
            if ($form->get('file')->getData()) {
                $bannerUpload = $this->get('wi.admin.banner.upload');
                $filename = $bannerUpload->upload(
                    $form->get('file')->getData(),
                    [
                        'crop_x' => (int) $form->get('crop_x')->getData(),
                        'crop_y' => (int) $form->get('crop_y')->getData(),
                        'crop_height' => (int) $form->get('crop_height')->getData(),
                        'crop_width' => (int) $form->get('crop_width')->getData(),
                        'cropSave' => (bool) $form->get('cropSave')->getData(),
                    ]
                );
                // Set filename.
                $banner->setImageUrl($filename);
            }

            // Zapis.
            $em->persist($banner);
            $em->flush();

            return $this->redirectToRoute('wi_admin_banner_show', [
                'id' => $banner->getId(),
            ]);
        }

        return $this->render('WiAdminBannerBundle:Banner:new.html.php', [
            'banner' => $banner,
            'form' => $form->createView(),
        ]);
    }


    /**
     * Finds and displays a banner entity.
     *
     * @param Banner $banner
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Banner $banner)
    {
        return $this->render('WiAdminBannerBundle:Banner:show.html.php', [
            'banner' => $banner,
        ]);
    }

    /**
     * Displays a form to edit an existing banner entity.
     *
     * @param Request $request
     * @param Banner $banner
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Banner $banner)
    {
        // Generowanie formularza.
        $editForm = $this->createForm(BannerType::class, $banner);
        $editForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($editForm->isSubmitted() && $editForm->isValid()) {
            // Upload file.
            if ($editForm->get('file')->getData()) {
                $bannerUpload = $this->get('wi.admin.banner.upload');
                $filename = $bannerUpload->upload(
                    $editForm->get('file')->getData(),
                    [
                        'crop_x' => (int) $editForm->get('crop_x')->getData(),
                        'crop_y' => (int) $editForm->get('crop_y')->getData(),
                        'crop_height' => (int) $editForm->get('crop_height')->getData(),
                        'crop_width' => (int) $editForm->get('crop_width')->getData(),
                        'cropSave' => (bool) $editForm->get('cropSave')->getData(),
                    ]
                );
                // Set filename.
                $banner->setImageUrl($filename);
            }

            // Zapis.
            $this->getDoctrine()->getManager()->flush();

            return $this->redirectToRoute('wi_admin_banner_show', [
                'id' => $banner->getId(),
            ]);
        }

        return $this->render('WiAdminBannerBundle:Banner:edit.html.php', [
            'banner' => $banner,
            'edit_form' => $editForm->createView(),
        ]);
    }

    /**
     * Deletes a question entity.
     *
     * @param Request $request
     * @param Banner $banner
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, Banner $banner)
    {
        // Generowanie formularza.
        $deleteForm = $this->createForm(BannerDeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usunięcie.
            $em->remove($banner);
            $em->flush();

            return $this->redirectToRoute('wi_admin_banner_index');
        }

        return $this->render('WiAdminBannerBundle:Banner:delete.html.php', [
            'banner' => $banner,
            'deleteForm' => $deleteForm->createView(),
        ]);
    }

    /**
     * Update a order banner entity.
     *
     * @param Banner $banner
     * @param string $sort_dir
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function updateOrderAction(Banner $banner, $sort_dir)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        $bannerRepo = $em->getRepository('WiAdminBannerBundle:Banner');
        $banners = $bannerRepo->findAllActive();
        $bannerSet = $em->getRepository('WiAdminBannerBundle:BannerSet')
            ->findAll()
        ;

        // Poprawienie kolejności. Może wystąpić sytuacja, że kilka kategorii
        // będzie miała przypisany ten sam numer w kolejności i wtedy
        // przenoszenie o '1' nie spełnia swojej roli.
        $fixOrder = function ($bannerSet, $banners) use (&$fixOrder, $em) {
            foreach($bannerSet as $set) {
                $order = 1;
                foreach($banners as $banner) {
                    if($set->getId()==$banner->getBannerSet()->getId()) {
                        $banner->setOrderNr($order++);
                        $em->persist($banner);
                        $em->flush();
                    }
                }
            }
        };

        $fixOrder($bannerSet,$banners);

        switch ($sort_dir) {
            case 'up': // Przesunięcie w górę.
                $nextTo = $bannerRepo->findOneBy([
                    'orderNr' => $banner->getOrderNr() - 1
                ]);

                if (! is_null($nextTo)) {
                    $nextTo->setOrderNr($banner->getOrderNr());
                    $banner->setOrderNr($banner->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            case 'down': // Przesunięcie w dół.
                $nextTo = $bannerRepo->findOneBy([
                    'orderNr' => $banner->getOrderNr() + 1
                ]);

                if (! is_null($nextTo)) {
                    $banner->setOrderNr($nextTo->getOrderNr());
                    $nextTo->setOrderNr($nextTo->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            default:
                break;
        }

        $em->persist($banner);
        $em->flush();

        return $this->redirectToRoute('wi_admin_banner_index');
    }
}
