<?php

namespace Wi\Admin\BlockBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\BlockBundle\Entity\ItemHTML;
use Wi\Admin\BlockBundle\Form\ItemHTMLType;
use Wi\Admin\BlockBundle\Form\ItemHTMLDeleteType;

/**
 * ItemHTML controller.
 *
 * @author Maksymilian Baryczka <mmaxioo33@gmail.com>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class ItemHTMLController extends Controller
{
    /**
     * List all block and item entities.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction()
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich bloków.
        $blocks = $em->getRepository('WiAdminBlockBundle:Block')->findAll();

        return $this->render('WiAdminBlockBundle:ItemHTML:index.html.php', [
            'blocks' => $blocks,
        ]);
    }

    /**
     * Creates a new itemHTML entity.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     * @throws NotFoundHttpException
     */
    public function newAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie nowego obiektu strony.
        $itemHTML = new ItemHTML();

        // Dodanie domyślnego bloku, jeśli został przekazany oraz istnieje.
        if ($request->query->getInt('block_id')) {
            $block = $em->getRepository('WiAdminBlockBundle:Block')
                ->findOneById($request->query->getInt('block_id'))
            ;

            if (is_null($block)) {
                throw $this->createNotFoundException();
            }

            $itemHTML->setBlock($block);
        }

        // Generowanie formularza.
        $form = $this->createForm(ItemHTMLType::class, $itemHTML);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Zapis.
            $em->persist($itemHTML);
            $em->flush();

            return $this->redirectToRoute('wi_admin_block_itemHTML_show', [
                'id' => $itemHTML->getId(),
            ]);
        }

        return $this->render('WiAdminBlockBundle:ItemHTML:new.html.php', [
            'form' => $form->createView(),
            'itemHTML' => $itemHTML,
        ]);
    }

    /**
     * Displays a form to edit an existing page entity.
     *
     * @param Request $request
     * @param ItemHTML $itemHTML
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, ItemHTML $itemHTML)
    {
        // Generowanie formularza.
        $editForm = $this->createForm(ItemHTMLType::class, $itemHTML);
        $editForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($editForm->isSubmitted() && $editForm->isValid()) {
            // Zapis.
            $this->getDoctrine()->getManager()->flush();

            return $this->redirectToRoute('wi_admin_block_itemHTML_show', [
                'id' => $itemHTML->getId(),
            ]);
        }

        return $this->render('WiAdminBlockBundle:ItemHTML:edit.html.php', [
            'edit_form' => $editForm->createView(),
            'itemHTML' => $itemHTML,
        ]);
    }

    /**
     * Finds and displays a item entity.
     *
     * @param ItemHTML $itemHTML
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(ItemHTML $itemHTML)
    {
        return $this->render('WiAdminBlockBundle:ItemHTML:show.html.php', [
            'itemHTML' => $itemHTML,
        ]);
    }

    /**
     * Deletes a item entity.
     *
     * @param Request $request
     * @param ItemHTML $itemHTML
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, ItemHTML $itemHTML)
    {
        // Generowanie formularza.
        $deleteForm = $this->createForm(ItemHTMLDeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usunięcie.
            $em->remove($itemHTML);
            $em->flush();

            return $this->redirectToRoute('wi_admin_block_itemHTML_index');
        }

        return $this->render('WiAdminBlockBundle:ItemHTML:delete.html.php', [
            'deleteForm' => $deleteForm->createView(),
            'itemHTML' => $itemHTML,
        ]);
    }

    /**
     * Update a order item entity.
     *
     * @param ItemHTML $ItemHTML
     * @param string $sort_dir
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function updateOrderAction(ItemHTML $itemHTML, $sort_dir)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $blockRepo = $em->getRepository('WiAdminBlockBundle:Block');
        $itemRepo = $em->getRepository('WiAdminBlockBundle:ItemHTML');

        // Pobranie bloków.
        $blocks = $blockRepo->findAll();

        // Poprawienie kolejności. Może wystąpić sytuacja, że kilka bloków
        // i elementów będzie miało przypisany ten sam numer w kolejności
        // i wtedy przenoszenie o '1' nie spełnia swojej roli.
        $fixOrder = function ($blocks, $order = 1) use ($em) {
            foreach($blocks as $block) {
                if ($block->getItems()->count() > 0) {
                    if (isset($item) && $block->getId() != $item->getBlock()->getId()) {
                        $order = 1;
                    }

                    foreach ($block->getItems() as $item) {
                        $item->setOrderNr($order++);
                        $em->persist($item);
                    }
                }

                $em->flush();
            }
        };

        $fixOrder($blocks);

        switch ($sort_dir) {
            case 'up': // Przesunięcie w górę.
                $nextTo = $itemRepo->findOneBy([
                    'orderNr' => $itemHTML->getOrderNr() - 1,
                    'block' => $itemHTML->getBlock(),
                ]);

                if (! is_null($nextTo)) {
                    $nextTo->setOrderNr($itemHTML->getOrderNr());
                    $itemHTML->setOrderNr($itemHTML->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            case 'down': // Przesunięcie w dół.
                $nextTo = $itemRepo->findOneBy([
                    'orderNr' => $itemHTML->getOrderNr() + 1,
                    'block' => $itemHTML->getBlock(),
                ]);

                if (! is_null($nextTo)) {
                    $itemHTML->setOrderNr($nextTo->getOrderNr());
                    $nextTo->setOrderNr($nextTo->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            default:
                break;
        }

        $em->persist($itemHTML);
        $em->flush();

        return $this->redirectToRoute('wi_admin_block_itemHTML_index');
    }
}
