<?php

namespace Wi\Admin\ContactBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\ContactBundle\Entity\Answer;
use Wi\Admin\ContactBundle\Entity\AnswerNote;
use Wi\Admin\ContactBundle\Entity\Form;
use Wi\Admin\NewsLetterBundle\Service\RecipientManager;

/**
 * Inbox Controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class InboxController extends Controller
{
    /**
     * Skrzynka odbiorcza dla formularzy kontaktowych.
     *
     * @param Request $request
     * @param Form $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function messagesAction(Request $request, Form $form = null)
    {
        $forms = $this->getFormRepo()->findByFormType(Form::TYPE_CONTACT);
        $statuses = $this->getAnswerStatusRepo()->findAll();

        if (is_null($form)) {
            $answers = $this->getAnswerRepo()->findByFormType(Form::TYPE_CONTACT);
        } else {
            $answers = $this->getAnswerRepo()->findByForm($form);
        }


        // Przetworzenie obiektu status na tablicę.
        $statusesTable = [];

        foreach ($statuses as $key => $statusItem) {
            $statusesTable[$statusItem->getGroup()][$statusItem->getId()] = $statusItem->getName();
        }

        return $this->render('WiAdminContactBundle:Inbox:messages.html.php', [
            'answers' => $answers,
            'form' => $form,
            'forms' => $forms,
            'statusesTable' => $statusesTable,
        ]);
    }

    /**
     * Skrzynka odbiorcza dla formularzy leadów.
     *
     * @param Request $request
     * @param Form|null $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function leadsAction(Request $request, Form $form = null)
    {
        // Pobranie odpowiedzi.
        if (is_null($form)) {
            $answers = $this->getAnswerRepo()->findByFormType(Form::TYPE_LEAD);
        } else {
            $answers = $this->getAnswerRepo()->findByForm($form);
        }

        $forms = $this->getFormRepo()->findByFormType(Form::TYPE_LEAD);
        $statuses = $this->getAnswerStatusRepo()->findAll();

        // Przetworzenie obiektu status na tablicę.
        $statusesTable = [];

        foreach ($statuses as $key => $statusItem) {
            $statusesTable[$statusItem->getGroup()][$statusItem->getId()] = $statusItem->getName();
        }

        $recipientGroups = $this->getRecipientGroupRepo()->findAll();

        return $this->render('WiAdminContactBundle:Inbox:leads.html.php', [
            'answers' => $answers,
            'statusesTable' => $statusesTable,
            'form' => $form,
            'forms' => $forms,
            'recipientGroups' => $recipientGroups,
        ]);
    }

    /**
     * Aktualizacja statusu odpowiedzi.
     *
     * @param Request $request
     * @param Answer $answer
     * @return RedirectResponse
     */
    public function changeStatusAction(Request $request, Answer $answer)
    {
        // Utworzenie instancji Entity Menagera
        $em = $this->getDoctrine()->getManager();
        $status = $this->getAnswerStatusRepo()
            ->find($request->request->getInt('status'))
        ;

        if (! is_null($status)) {
            $answer->setStatus($status);
            $em->persist($answer);
            $em->flush($answer);
        }

        return $this->redirect($request->headers->get('referer'));
    }

    /**
     * Zapis notatki dla odpowiedzi.
     *
     * @param Request $request
     * @param Answer $answer
     * @return RedirectResponse
     */
    public function saveNoteAction(Request $request, Answer $answer)
    {
        $answerNote = new AnswerNote();
        $answerNote->setNote($request->request->get('note'));
        $answer->addNote($answerNote);

        // Utworzenie instancji Entity Menagera
        $em = $this->getDoctrine()->getManager();
        $em->persist($answer);
        $em->flush($answer);

        return $this->redirect($request->headers->get('referer'));
    }

    public function moveToNewsletterAction(Request $request, RecipientManager $manager)
    {
        $arr = [
            'email' => trim(strip_tags($request->request->get('email'))),
            'name' => null,
            'group' => $request->request->getInt('recipient_group_id'),
        ];

        if (! $manager->checkExist($arr)) {
            $manager->signUpRecipient($arr);

            $recipient = $manager->getRecipient();

            return $this->redirectToRoute('wi_admin_newsletter_recipients_edit', [
                'id' => $recipient->getId(),
            ]);
        }

        $translator = $this->get('translator');

        $this->addFlash(
            'admin.recipient_exist',
            $translator->trans('contactLeads.recipientAlreadyExist', [
                '%email%' => $arr['email'],
                '%group%' => $manager->getRecipient()->getRecipientGroup()->getName(),
            ])
        );

        return $this->redirect($request->headers->get('referer'));
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Pobiera repozytorium dla odpowiedzi.
     *
     * @return AnswerRepository
     */
    private function getAnswerRepo()
    {
        return $this->getDoctrine()->getManager()
            ->getRepository('WiAdminContactBundle:Answer')
        ;
    }

    /**
     * Pobiera repozytorium dla statusów odpowiedzi.
     *
     * @return AnswerStatusRepository
     */
    private function getAnswerStatusRepo()
    {
        return $this->getDoctrine()->getManager()
            ->getRepository('WiAdminContactBundle:AnswerStatus')
        ;
    }

    /**
     * Pobiera repozytorium dla formularzy.
     *
     * @return FormRepository
     */
    private function getFormRepo()
    {
        return $this->getDoctrine()->getManager()
            ->getRepository('WiAdminContactBundle:Form')
        ;
    }

    /**
     * Pobiera repozytorium dla odbiorców formularzy.
     *
     * @return RecipientGroupRepository
     */
    private function getRecipientGroupRepo()
    {
        return $this->getDoctrine()->getManager()
            ->getRepository('WiAdminNewsLetterBundle:RecipientGroup')
        ;
    }
}
