<?php

namespace Wi\Admin\ContactBundle\Repository;

use Doctrine\ORM\EntityRepository;
use Wi\Admin\ContactBundle\Entity\Form;

/**
 * AnswerRepository
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class AnswerRepository extends EntityRepository
{
    /**
     * ID statusu: New.
     *
     * @const STATUS_NEW
     */
    const STATUS_NEW = 1;

    /**
     * Liczba wiadomości na podstawie przekazanego formularza.
     *
     * @param Form $form
     * @return int
     */
    public function countAllByForm(Form $form = null)
    {
        $qb = $this->createQueryBuilder('a');

        return (int) $qb
            ->select($qb->expr()->count('a'))
            ->where('a.form = :form')
            ->setParameters([
                'form' => $form,
            ])
            ->getQuery()
            ->getSingleScalarResult()
        ;
    }

    /**
     * Pobiera liczbę nieprzeczytanych wiadomości na podstawie
     * przekazanego formularza.
     *
     * @param Form $form
     * @return int
     */
    public function countUnreadedByForm(Form $form = null)
    {
        $qb = $this->createQueryBuilder('a');

        return (int) $qb
            ->select($qb->expr()->count('a'))
            ->innerJoin('a.status', 's')
            ->where('a.form = :form')
            ->andWhere('s.id = :status_id')
            ->setParameters([
                'form' => $form,
                'status_id' => self::STATUS_NEW,
            ])
            ->getQuery()
            ->getSingleScalarResult()
        ;
    }

    /**
     * Pobiera odpowiedzi na podstawie formularza posortowane
     * według daty malejąco.
     *
     * @param Form $form
     * @return array
     */
    public function findByForm(Form $form)
    {
        return $this->createQueryBuilder('a')
            ->where('a.form = :form')
            ->setParameters([
                'form' => $form,
            ])
            ->orderBy('a.dateCreated', 'DESC')
            ->getQuery()
            ->getResult()
        ;
    }

    /**
     * Pobiera odpowiedzi na podstawie typu formularza oraz posortowane
     * według daty malejąco.
     *
     * @param string $type
     * @return array
     */
    public function findByFormType($type)
    {
        return $this->createQueryBuilder('a')
            ->innerJoin('a.form', 'f')
            ->where('f.type = :type')
            ->setParameters([
                'type' => $type,
            ])
            ->orderBy('a.dateCreated', 'DESC')
            ->getQuery()
            ->getResult()
        ;
    }

    /**
     * Pobiera liczbę odpowiedzi dla podanego typu.
     *
     * @param string $type
     * @return int
     */
    public function countByFormType($type)
    {
        $qb = $this->createQueryBuilder('a');

        return (int) $qb
            ->select($qb->expr()->count('a'))
            ->innerJoin('a.form', 'f')
            ->where('f.type = :type')
            ->setParameters([
                'type' => $type,
            ])
            ->getQuery()
            ->getSingleScalarResult()
            ;
    }

    /**
     * Pobiera liczbę nieprzeczytanych wiadomości dla podanego typu.
     *
     * @param string $type
     * @return int
     */
    public function countUnreadedByFormType($type)
    {
        $qb = $this->createQueryBuilder('a');

        return (int) $qb
            ->select($qb->expr()->count('a'))
            ->innerJoin('a.status', 's')
            ->innerJoin('a.form', 'f')
            ->where('f.type = :type')
            ->andWhere('s.id = :status_id')
            ->setParameters([
                'type' => $type,
                'status_id' => self::STATUS_NEW,
            ])
            ->getQuery()
            ->getSingleScalarResult()
            ;
    }

    /**
     * Pobiera liczbę wszystkich wiadomości dla każdego formularza
     * i zwraca w formie tablicy, gdzie kluczem jest ID formularza
     * a wartością liczba wszystkich wiadomości.
     *
     * @return array
     */
    public function countGroupByForm()
    {
        $qb = $this->createQueryBuilder('a');
        $results = $qb
            ->select($qb->expr()->count('a') . ' AS liczba')
            ->addSelect('f.id AS form_id')
            ->innerJoin('a.form', 'f')
            ->groupBy('a.form')
            ->getQuery()
            ->getResult()
        ;
        $out = [];$forms = $this->_em->getRepository('WiAdminContactBundle:Form')->findAll();

        foreach ($forms as $form) {
            $out[$form->getId()] = 0;
        }

        foreach ($results as $result) {
            $out[$result['form_id']] = $result['liczba'];
        }

        return $out;
    }

    /**
     * Pobiera liczbę nieprzeczytanych wiadomości dla kazdego formularza
     * i zwraca w formie tablicy, gdzie kluczem jest ID formularza
     * a wartością liczba nieprzeczytanych wiadomości.
     *
     * @return array
     */
    public function countUnreadedGroupByForm()
    {
        $qb = $this->createQueryBuilder('a');
        $results = $qb
            ->select($qb->expr()->count('a') . ' AS liczba')
            ->addSelect('f.id AS form_id')
            ->innerJoin('a.form', 'f')
            ->innerJoin('a.status', 's')
            ->where('s.id = :status_id')
            ->groupBy('a.form')
            ->setParameters([
                'status_id' => self::STATUS_NEW,
            ])
            ->getQuery()
            ->getResult()
        ;
        $out = [];
        $forms = $this->_em->getRepository('WiAdminContactBundle:Form')->findAll();

        foreach ($forms as $form) {
            $out[$form->getId()] = 0;
        }

        foreach ($results as $result) {
            $out[$result['form_id']] = (int) $result['liczba'];
        }

        return $out;
    }
}
