<?php

namespace Wi\Admin\CoreBundle\Service;

use Symfony\Bundle\FrameworkBundle\Templating\EngineInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Translation\TranslatorInterface;
use Wi\Admin\CoreBundle\Service\Config;

/**
 * Mailer.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class Mailer
{
    /**
     * @var string
     */
    private $transport;

    /**
     * @var string
     */
    private $host;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $password;

    /**
     * @var int
     */
    private $port;

    /**
     * Nazwa pod jaką wysyłane są maile, np. 'John Doe'.
     *
     * @var string
     */
    private $name;

    /**
     * @var EngineInterface
     */
    private $templating;

    /**
     * @var TranslatorInterface
     */
    protected $translator;

    /**
     * @var \Swift_Mailer
     */
    protected $mailer;

    /**
     * @var \Swift_Message
     */
    protected $message;


    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Constructor.
     *
     * @param Config $config
     * @param EngineInterface $templating
     * @param \Swift_Mailer $mailer
     * @param TranslatorInterface $translator
     */
    public function __construct(Config $config, ContainerInterface $container, EngineInterface $templating, \Swift_Mailer $mailer, TranslatorInterface $translator)
    {
        $this->username = $container->getParameter('mailer_user');

        $this->templating = $templating;
        $this->translator = $translator;

        // Initialize functions.
        $this->setMailer($mailer);
        $this->setMessage();
    }

    /**
     * Wysłanie wiadomości.
     *
     * @return int
     */
    public function send()
    {
        return $this->mailer->send($this->message);
    }

    // ------------------------------------------------------------
    // Protected functions.
    // ------------------------------------------------------------

    /**
     * Wygenerowanie widoku treści wiadomości.
     *
     * @param string $view
     * @param array $params
     * @return \Symfony\Component\HttpFoundation\Response
     */
    protected function render($view, $params = [])
    {
        return $this->templating->render($view, array_merge($params, [
            'message' => $this->message,
            'subject' => $this->message->getSubject(),
        ]));
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Ustawienie mailera.
     *
     * @param \Swift_Mailer $mailer
     * @return Mailer
     */
    private function setMailer($mailer)
    {
        $this->mailer = $mailer;

        return $this;
    }

    /**
     * Wstępne ustawienie parametrów wiadomości.
     */
    private function setMessage()
    {
        $from = $this->username;

        if (! empty($this->name)) {
            $from = [
                $this->username => $this->name,
            ];
        }

        $this->message = (new \Swift_Message())
            ->setFrom($from)
        ;
    }
}
