<?php

namespace Wi\Admin\CoreBundle\Service;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;
use Symfony\Component\Translation\TranslatorInterface;
use Wi\Admin\CoreBundle\Entity\Menu;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Admin\CoreBundle\Cache\CacheMenuManager;

/**
 * MenuAdmin manager.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class MenuAdmin
{
    /**
     * @var AuthorizationCheckerInterface
     */
    private $authChecker;

    /**
     * @var CacheMenuManager
     */
    private $cacheManager;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var Translator
     */
    private $translator;

    /**
     * Constructor.
     *
     * @param AuthorizationCheckerInterface $authChecker
     * @param Config $config
     * @param EntityManagerInterface $em
     * @param RouterInterface $router
     * @param TranslatorInterface $translator
     */
    public function __construct(
        AuthorizationCheckerInterface $authChecker,
        CacheMenuManager $cacheManager,
        Config $config,
        EntityManagerInterface $em,
        RouterInterface $router,
        TranslatorInterface $translator
    ) {
        $this->authChecker = $authChecker;
        $this->cacheManager = $cacheManager;
        $this->config = $config;
        $this->em = $em;
        $this->router = $router;
        $this->translator = $translator;
    }

    // -------------------------------------------------------------------------

    /**
     * Generuje strukturę HTML Menu
     *
     * @param int $group_id
     * @return string
     */
    public function getMenu(int $group_id, $attr, $menuIsActive) : string
    {
        // Wybranie grupy menu.
        $this->cacheManager->setItemGroup($group_id);

        if (! $this->cacheManager->isHit()) { // Sprawdzenie, czy nie ma cache.
            $menu = $this->em->getRepository('WiAdminCoreBundle:Menu')
                ->findAllWithoutParent($group_id)
            ;

            // Zapisanie cache.
            $this->cacheManager->set($menu);
        } else { // Jeśli cache jest.
            $menu = $this->cacheManager->get();
        }

        return $this->generateMenu($menu, $attr, true, '', $menuIsActive);
    }

    // -------------------------------------------------------------------------

    private function generateMenu(
        $arr,
        $ulAttr = 'id="menu-showhide" class="topnav slicknav vertical menu',
        $first = true,
        $html = '',
        $menuIsActive = ''
    ) {
        $html .= '<ul ' . $ulAttr . '>';

        foreach ($arr as $item) {
            if ($this->authChecker->isGranted($item->getPermission())) {
                if ($item->getId() == 270 && ! $this->config->isWicmsCommentsEnabled()) {
                    continue;
                }

                $link = $item->getEnableLink() ? $item->getLink() : $this->router->generate($item->getRouteName(), $item->getRouteParams());
                $active = $link == $menuIsActive;
                $html .= '<li ' . ($active ? 'class="active"' : '') . '>';
                $html .= '<a href="' . $link . '"><i class="' . $item->getClass() . '"></i>';

                if ($first) {
                    $html .= '<span>';
                }

                $html .= $this->translator->trans($item->getName());

                if ($first) {
                    $html .= '</span>';
                }

                $html .= '</a>';

                if ($item->getChildren()->count() > 0) {
                    $html .= $this->generateMenu($item->getChildren(), '', false, '', $menuIsActive);
                }

                $html .= '</li>';
            }
        }

        $html .= '</ul>';

        return $html;
    }
}
