<?php

namespace Wi\Admin\CoreBundle\Service\Seo;

use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RouterInterface;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Admin\PageBundle\Entity\Page;
use Wi\Admin\CoreBundle\Service\Seo\GenerateTypeInterface;
use Wi\Admin\CoreBundle\Service\Seo\LengthStatusInterface;
use Wi\Front\CoreBundle\Utils\Slugger;

/**
 * Seo generator for pages.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class SeoPage
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * Constructor.
     *
     * @param Config $config
     * @param RouterInterface $router
     * @param Slugger $slugger
     */
    public function __construct(Config $config, RouterInterface $router, Slugger $slugger)
    {
        $this->config = $config;
        $this->router = $router;
        $this->slugger = $slugger;
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca gotowe metadatane dla strony.
     *
     * @param Page $page
     * @return string
     */
    public function getMeta(Page $page)
    {
        $arr = [
            $this->metaTitleRender($page),
            $this->metaDescriptionRender($page),
            $this->metaKeywordsRender($page),
            $this->canonicalRender($page),
            $this->ogRender($page),
        ];

        return implode(PHP_EOL.'    ', $arr).PHP_EOL;
    }

    /**
     * Zwraca tablicę wygenerowanych metadanych dla strony wraz ze sposobem
     * generowania poszczególnych danych.
     *
     * @param Page $page
     * @return array
     */
    public function getMetaArray(Page $page)
    {
        $title = $this->getTitleGenerate($page);
        $desc = $this->getDescriptionGenerate($page);
        $keywords = $this->getKeywordsGenerate($page);
        $canonical = $this->getCanonicalGenerate($page);
        $image = $this->getImageGenerate($page);

        return [
            'title' => [
                'type' => $title['type'],
                'value' => $title['value'],
                'length' => $this->getTitleLength($page),
            ],
            'description' => [
                'type' => $desc['type'],
                'value' => $desc['value'],
                'length' => $this->getDescriptionLength($page),
            ],
            'keywords' => [
                'type' => $keywords['type'],
                'value' => $keywords['value'],
            ],
            'canonical' => [
                'type' => $canonical['type'],
                'value' => $canonical['value'],
            ],
            'image' => [
                'type' => $image['type'],
                'value' => $image['value'],
            ],
        ];
    }

    /**
     * Zwraca tablicę metadanych dla strony.
     *
     * @param Page $page
     * @return array
     */
    public function getMetaArrayValues(Page $page)
    {
        $out = [];

        foreach ($this->getMetaArray($page) as $key => $value) {
            $out[$key] = $value['value'];
        }

        return $out;
    }

    /**
     * Zwraca tablicę wygenerowanych danych OpenGraph dla strony wraz ze sposobem
     * generowania poszczególnych danych.
     *
     * @param Page $page
     * @return array
     */
    public function getOgArray(Page $page)
    {
        $title = $this->getTitleGenerate($page);
        $desc = $this->getDescriptionGenerate($page);
        $image = $this->getImageGenerate($page);
        $type = $this->getOgTypeGenerate($page);

        return [
            'og_title' => [
                'type' => $title['type'],
                'value' => $title['value'],
            ],
            'og_description' => [
                'type' => $desc['type'],
                'value' => $desc['value'],
            ],
            'og_type' => [
                'type' => $type['type'],
                'value' => $type['value'],
            ],
            'og_image' => [
                'type' => $image['type'],
                'value' => $image['value'],
            ],
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca tablicę z wygenerowanym kanonicznym adresem URL dla strony oraz
     * sposobem generowania.
     *
     * @param Page $page
     * @return array
     */
    public function getCanonicalGenerate(Page $page)
    {
        if (boolval(intval($this->config->get('metadata.canonical')))) {
            if (! empty($page->getMetaCanonical())) {
                if (! boolval(intval($this->config->get('metadata.canonicalForce')))) {
                    return [
                        'type' => GenerateTypeInterface::MANUAL,
                        'value' => $page->getMetaCanonical(),
                    ];
                }
            }
        }

        return [
            'type' => GenerateTypeInterface::DYNAMIC,
            'value' => $this->router->generate(
                'wi_front_page_index',
                [
                    'id' => $page->getId(),
                    'slug' => (! empty($page->getTitle()) ? $this->slugger->slugify($page->getTitle()) : '-'),
                ],
                UrlGeneratorInterface::ABSOLUTE_URL
            ),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym opisem strony oraz sposobem generowania.
     *
     * @param Page $page
     * @return array
     */
    public function getDescriptionGenerate(Page $page)
    {
        if (! empty($page->getMetaDescription())) {
            return [
                'type' => GenerateTypeInterface::MANUAL,
                'value' => htmlentities($page->getMetaDescription()),
            ];
        }

        return [
            'type' => GenerateTypeInterface::DEFAULT,
            'value' => htmlentities($this->config->get('metadata.description')),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym zdjęciem OpenGraph strony oraz
     * sposobem generowania.
     *
     * @param Page $page
     * @return array
     */
    public function getImageGenerate(Page $page)
    {
        $openGraphEnable = boolval(intval($this->config->get('metadata.openGraphEnable')));

        if ($openGraphEnable) {
            if (! empty($page->getMetaImage())) {
                return [
                    'type' => GenerateTypeInterface::MANUAL,
                    'value' => $page->getMetaImage(),
                ];
            }

            if (! empty($img = $this->config->get('metadata.openGraphDefaultImage'))) {
                return [
                    'type' => GenerateTypeInterface::DEFAULT,
                    'value' => $img,
                ];
            }
        }

        return [
            'type' => GenerateTypeInterface::EMPTY,
            'value' => null,
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanymi słowami kluczowymi dla strony oraz
     * sposobem generowania.
     *
     * @param Page $page
     * @return array
     */
    public function getKeywordsGenerate(Page $page)
    {
        $keywordsForAllPages = boolval(intval($this->config->get('metadata.keywordsForAllPages')));
        $affix = ! empty($page->getMetaKeywords()) ? ', ' : '';
        $affix .= $keywordsForAllPages ? $this->config->get('metadata.keywords') : '';

        if (boolval(intval($this->config->get('metadata.keywordsEnable')))) {
            if (! empty($page->getMetaKeywords())) {
                return [
                    'type' => GenerateTypeInterface::MANUAL,
                    'value' => $page->getMetaKeywords().$affix,
                ];
            }

            return [
                'type' => GenerateTypeInterface::DEFAULT,
                'value' => $page->getMetaKeywords().$affix,
            ];
        }

        return [
            'type' => GenerateTypeInterface::EMPTY,
            'value' => null,
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym typem OpenGraph dla strony oraz
     * sposobem generowania.
     *
     * @return array
     */
    public function getOgTypeGenerate(Page $page)
    {
        return [
            'type' => GenerateTypeInterface::DYNAMIC,
            'value' => $this->config->get('og.pageType'),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym tytułem strony oraz sposobem generowania.
     *
     * @param Page $page
     * @return array
     */
    public function getTitleGenerate(Page $page)
    {
        $titleForAllPages = boolval(intval($this->config->get('metadata.titleForAllPages')));
        $affix = $titleForAllPages ? $this->config->get('metadata.titleSeparator').$this->config->get('metadata.title') : '';

        if (! empty($page->getMetaTitle())) {
            return [
                'type' => GenerateTypeInterface::MANUAL,
                'value' => htmlentities($page->getMetaTitle().$affix),
            ];
        }

        if (! empty($page->getTitle())) {
            return [
                'type' => GenerateTypeInterface::DYNAMIC,
                'value' => htmlentities($page->getTitle().$affix),
            ];
        }

        return [
            'type' => GenerateTypeInterface::DEFAULT,
            'value' => htmlentities($this->config->get('metadata.title')),
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje tytuł strony.
     *
     * @param Page $page
     * @return string
     */
    public function metaTitleRender(Page $page)
    {
        return sprintf('<title>%s</title>', $this->getTitleGenerate($page)['value']);
    }

    /**
     * Renderuje opis strony.
     *
     * @param Page $page
     * @return string
     */
    public function metaDescriptionRender(Page $page)
    {
        return sprintf(
            '<meta name="description" content="%s">',
            $this->getDescriptionGenerate($page)['value']
        );
    }

    /**
     * Renderuje słowa kluczowe strony.
     *
     * @param Page $page
     * @return string
     */
    public function metaKeywordsRender(Page $page)
    {
        if (boolval(intval($this->config->get('metadata.keywordsEnable')))) {
            return sprintf(
                '<meta name="keywords" content="%s">',
                $this->getKeywordsGenerate($page)['value']
            );
        }

        return;
    }

    /**
     * Renderuje kanoniczny adres URL.
     *
     * @param Page $page
     * @return string
     */
    public function canonicalRender(Page $page)
    {
        if (boolval(intval($this->config->get('metadata.canonical')))) {
            return sprintf(
                '<link rel="canonical" href="%s">',
                $this->getCanonicalGenerate($page)['value']
            );
        }

        return;
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje OpenGraph.
     *
     * @param Page $page
     * @return string
     */
    public function ogRender(Page $page)
    {
        if (boolval(intval($this->config->get('metadata.openGraphEnable')))) {
            $arr = [
                $this->ogSiteNameRender($page),
                $this->ogTitleRender($page),
                $this->ogDescriptionRender($page),
                $this->ogTypeRender($page),
                $this->ogUrlRender($page),
                $this->ogImageRender($page),
            ];

            return trim(implode(PHP_EOL.'    ', $arr).PHP_EOL);
        }

        return;
    }

    /**
     * Renderuje nazwę strony dla OpenGraph.
     *
     * @param Page $page
     * @return string
     */
    public function ogSiteNameRender(Page $page)
    {
        return sprintf(
            '<meta property="og:site_name" content="%s">',
            $this->getTitleGenerate($page)['value']
        );
    }

    /**
     * Renderuje tytuł dla OpenGraph.
     *
     * @param Page $page
     * @return string
     */
    public function ogTitleRender(Page $page)
    {
        return sprintf(
            '<meta property="og:title" content="%s">',
            $this->getTitleGenerate($page)['value']
        );
    }

    /**
     * Renderuje opis strony dla OpenGraph.
     *
     * @param Page $page
     * @return string
     */
    public function ogDescriptionRender(Page $page)
    {
        return sprintf(
            '<meta property="og:description" content="%s">',
            $this->getDescriptionGenerate($page)['value']
        );
    }

    /**
     * Renderuje typ treści dla OpenGraph.
     *
     * @param Page $page
     * @return string
     */
    public function ogTypeRender(Page $page)
    {
        return sprintf(
            '<meta property="og:type" content="%s">',
            $this->getOgTypeGenerate($page)['value']
        );
    }

    /**
     * Renderuje adres URL dla OpenGraph.
     *
     * @param Page $page
     * @return string
     */
    public function ogUrlRender(Page $page)
    {
        return sprintf(
            '<meta property="og:url" content="%s">',
            $this->getCanonicalGenerate($page)['value']
        );
    }

    /**
     * Renderuje zdjęcie dla OpenGraph.
     *
     * @param Page $page
     * @return string
     */
    public function ogImageRender(Page $page)
    {
        if (! empty($img = $this->getImageGenerate($page)['value'])) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        if (! empty($img = $this->config->get('metadata.openGraphDefaultImage'))) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        return;
    }

    // -------------------------------------------------------------------------

    /**
     * Sprawdza, czy tytuł jest właściwej długości.
     *
     * @param Page $page
     * @return array
     */
    public function getTitleLength(Page $page)
    {
        $length = mb_strlen(html_entity_decode($this->getTitleGenerate($page)['value']));
        $from = intval($this->config->get('seo.metaTitleLengthFrom'));
        $to = intval($this->config->get('seo.metaTitleLengthTo'));

        if ($length > $to) {
            $status = LengthStatusInterface::TOO_LONG;
        } elseif ($length >= $from && $length <= $to) {
            $status = LengthStatusInterface::RIGHT;
        } else {
            $status = LengthStatusInterface::TOO_SHORT;
        }

        return [
            'length' => $length,
            'status' => $status,
        ];
    }

    /**
     * Sprawdza, czy opis jest właściwej długości.
     *
     * @param Page $page
     * @return array
     */
    public function getDescriptionLength(Page $page)
    {
        $length = mb_strlen(html_entity_decode($this->getDescriptionGenerate($page)['value']));
        $from = intval($this->config->get('seo.metaDescriptionLengthFrom'));
        $to = intval($this->config->get('seo.metaDescriptionLengthTo'));

        if ($length > $to) {
            $status = LengthStatusInterface::TOO_LONG;
        } elseif ($length >= $from && $length <= $to) {
            $status = LengthStatusInterface::RIGHT;
        } else {
            $status = LengthStatusInterface::TOO_SHORT;
        }

        return [
            'length' => $length,
            'status' => $status,
        ];
    }
}
