<?php

namespace Wi\Admin\CoreBundle\Service;

use Doctrine\ORM\EntityManagerInterface;
use Wi\Admin\CoreBundle\Service\Array2Xml;
use Wi\Admin\CoreBundle\Service\Seo;
use Wi\Admin\CoreBundle\Service\Utils;
use Wi\Front\CoreBundle\Utils\Slugger;

/**
 * Sitemap.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class Sitemap
{
    const XMLNS = 'http://www.sitemaps.org/schemas/sitemap/0.9';
    const XMLNS_IMAGE = 'http://www.google.com/schemas/sitemap-image/1.1';
    const PAGE_IDS = [];

    /**
     * @var Config
     */
    private $config;

    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var Seo
     */
    private $seo;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * @var Utils
     */
    private $utils;

    /**
     * @var string
     */
    private $lastmod;

    // -------------------------------------------------------------------------

    /**
     * @var array
     */
    private $params = [
        'multifile' => null,
        'splitPagesPerGroup' => null,
        'splitNewsPerCategory' => null,
        'splitNewsPerMonth' => null,
        'splitGalleriesPerAlbum' => null,
        'includePageImage' => null,
        'includeNewsImage' => null,
        'includeGalleryImages' => null,
        'showImagePublicIndex' => null,
    ];

    /**
     * @var string
     */
    private $web_dir;

    // -------------------------------------------------------------------------

    /**
     * Constructor.
     *
     * @param Config $config
     * @param EntityManagerInterface $em
     * @param Array2Xml $array2xml
     */
    public function __construct(Config $config, EntityManagerInterface $em, Seo $seo, Slugger $slugger, Utils $utils)
    {
        $this->config = $config;
        $this->em = $em;
        $this->seo = $seo;
        $this->slugger = $slugger;
        $this->utils = $utils;
        $this->web_dir = $config->getParameter('web_dir');

        $this->lastmod = (new \DateTime())->format('c');
        $this->params = [
            'multifile' => boolval(intval($this->config->get('sitemap.multifile'))),
            'splitPagesPerGroup' => boolval(intval($this->config->get('sitemap.splitPagesPerGroup'))),
            'splitNewsPerCategory' => boolval(intval($this->config->get('sitemap.splitNewsPerCategory'))),
            'splitNewsPerMonth' => boolval(intval($this->config->get('sitemap.splitNewsPerMonth'))),
            'splitGalleriesPerAlbum' => boolval(intval($this->config->get('sitemap.splitGalleriesPerAlbum'))),
            'includePageImage' => boolval(intval($this->config->get('sitemap.includePageImage'))),
            'includeNewsImage' => boolval(intval($this->config->get('sitemap.includeNewsImage'))),
            'includeGalleryImages' => boolval(intval($this->config->get('sitemap.includeGalleryImages'))),
            'showImagePublicIndex' => boolval(intval($this->config->get('sitemap.showImagePublicIndex'))),
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Generowanie pliku sitemap.xml w zależności od ustaionych
     * parametrów konfiguracyjnych.
     *
     * @return bool
     */
    public function generate()
    {
        // Usuniecie starych plików sitemap.
        array_map('unlink', glob($this->web_dir.'sitemap*.xml'));

        if ($this->params['multifile']) { // Wiele plików.
            return $this->generateMultiFile();
        } else { // Jeden plik.
            return $this->generateSingleFile();
        }
    }

    /**
     * Generowanie pojedynczego pliku sitemap.xml zawierającego wszystkie dane.
     *
     * @return bool
     */
    public function generateSingleFile()
    {
        $urls = array_merge(
            $this->prepareUrls($this->getManualPageSetArray(), $this->params['includePageImage']),
            $this->prepareUrls($this->getPagesArray(), $this->params['includePageImage']),
            $this->prepareUrls($this->getNewsArray(), $this->params['includeNewsImage'])
        );

        return (bool) $this->generateUrlsetFile($urls);
    }

    /**
     *
     *
     * @return bool
     */
    public function generateMultiFile()
    {
        $sitemaps = [];

        // Strony.
        if ($pages = $this->generatePagesFile()) {
            $sitemaps[] = [
                'sitemap' => [
                    'loc' => $this->utils->absoluteUrl($pages),
                    'lastmod' => $this->lastmod,
                ],
            ];
        }

        if ($news = $this->generateNewsFile()) {
            $sitemaps[] = [
                'sitemap' => [
                    'loc' => $this->utils->absoluteUrl($news),
                    'lastmod' => $this->lastmod,
                ],
            ];

        }

        return (bool) $this->generateSitemapIndexFile($sitemaps);
    }

    /**
     * Generowanie mapy dla aktualności.
     *
     * @return string
     */
    public function generateNewsFile()
    {
        if ($this->params['splitNewsPerCategory'] && $this->params['splitNewsPerMonth']) { // Dzielone kategorie i daty.
            $sitemapsParent = [];
            $i = 1;

            foreach ($this->getNewsCategoryArray() as $key => $category) {
                $sitemaps = [];

                foreach ($category['dates'] as $key2 => $date) {
                    $catArr = [];

                    if ($i == 1) {
                        $catArr = $this->prepareUrls([$this->seo->getMetaArrayValues($category['category'])], $this->params['includeNewsImage']);
                    }

                    $urls = array_merge(
                        $catArr,
                        $this->prepareUrls($date, $this->params['includeNewsImage'])
                    );

                    if ($month = $this->generateUrlsetFile($urls, 'aktualnosci-'.$category['category']->getName().'-'.$key2)) {
                        $sitemaps[] = [
                            'sitemap' => [
                                'loc' => $this->utils->absoluteUrl($month),
                                'lastmod' => $this->lastmod,
                            ],
                        ];
                    }

                    $i++;
                }

                if ($category = $this->generateSitemapIndexFile($sitemaps, 'aktualnosci-'.$category['category']->getName())) {
                    $sitemapsParent[] = [
                        'sitemap' => [
                            'loc' => $this->utils->absoluteUrl($category),
                            'lastmod' => $this->lastmod,
                        ],
                    ];
                }
            }

            return $this->generateSitemapIndexFile($sitemapsParent, 'aktualnosci');
        } elseif ($this->params['splitNewsPerCategory']) { // Dzielone na kategorie.
            $sitemaps = [];

            foreach ($this->getNewsCategoryArray() as $key => $category) {
                $urls = array_merge(
                    $this->prepareUrls([$this->seo->getMetaArrayValues($category['category'])], $this->params['includeNewsImage']),
                    $this->prepareUrls($category['news'], $this->params['includeNewsImage'])
                );

                if ($category = $this->generateUrlsetFile($urls, 'aktualnosci-'.$category['category']->getName())) {
                    $sitemaps[] = [
                        'sitemap' => [
                            'loc' => $this->utils->absoluteUrl($category),
                            'lastmod' => $this->lastmod,
                        ],
                    ];
                }
            }

            return $this->generateSitemapIndexFile($sitemaps, 'aktualnosci');
        } elseif ($this->params['splitNewsPerMonth']) { // Dzielone na daty.
            $sitemaps = [];

            foreach ($this->getNewsArray() as $key => $date) {
                $urls = array_merge(
                    $this->prepareUrls($date, $this->params['includeNewsImage'])
                );

                if ($month = $this->generateUrlsetFile($urls, 'aktualnosci-'.$key)) {
                    $sitemaps[] = [
                        'sitemap' => [
                            'loc' => $this->utils->absoluteUrl($month),
                            'lastmod' => $this->lastmod,
                        ],
                    ];
                }
            }

            return $this->generateSitemapIndexFile($sitemaps, 'aktualnosci');
        } else { // Nie dzielone.
            $urls = array_merge(
                $this->prepareUrls($this->getNewsArray(), $this->params['includeNewsImage'])
            );

            return $this->generateUrlsetFile($urls, 'aktualnosci');
        }
    }

    /**
     * Generowanie mapy dla stron.
     *
     * @return string
     */
    public function generatePagesFile()
    {
        if ($this->params['splitPagesPerGroup']) { // Dzielone na grupy.
            $sitemaps = [];

            foreach ($this->getPageGroupsArray() as $key => $group) {
                $urls = array_merge(
                    $this->prepareUrls($group['pages'], $this->params['includePageImage'])
                );

                if ($key == 1) {
                    $urls = array_merge(
                        $this->prepareUrls($this->getManualPageSetArray(), $this->params['includePageImage']),
                        $urls
                    );
                }

                // Strony.
                if ($group = $this->generateUrlsetFile($urls, 'artykuly-'.$group['group'])) {
                    $sitemaps[] = [
                        'sitemap' => [
                            'loc' => $this->utils->absoluteUrl($group),
                            'lastmod' => $this->lastmod,
                        ],
                    ];
                }
            }

            return $this->generateSitemapIndexFile($sitemaps, 'artykuly');
        } else { // Nie dzielone na grupy.
            $urls = array_merge(
                $this->prepareUrls($this->getManualPageSetArray(), $this->params['includePageImage']),
                $this->prepareUrls($this->getPagesArray(), $this->params['includePageImage'])
            );

            return $this->generateUrlsetFile($urls, 'artykuly');
        }
    }

    // -------------------------------------------------------------------------

    /**
     * Generuje plik "sitemapindex" na podstawie tablicy przekazanych sitemap
     * oraz o podanej nazwie, która jest przetwarzana na url. Zwraca fałsz,
     * jeśli niepowiedzie się generowanie pliku lub nazwę pliku w przypadku
     * prawidłowego wygenerowania pliku.
     *
     * @param array $sitemaps
     * @param string $fileAffix
     * @return bool|string
     */
    public function generateSitemapIndexFile($sitemaps, $fileAffix = '')
    {
        $arr = [
            'sitemapindex' => [
                'xmlns' => self::XMLNS,
            ],
        ];

        if (! empty($fileAffix)) {
            $fileAffix = '-' . $this->slugger->slugify($fileAffix);
        }

        $arr['sitemapindex'] = array_merge($arr['sitemapindex'], $sitemaps);

        try {
            $filename = 'sitemap'.$fileAffix.'.xml';
            $array2xml = new Array2Xml();
            $xml = $array2xml->parse($arr);
            $xml->save($this->web_dir.$filename);
        } catch (\Exception $e) {
            return false;;
        }

        return $filename;
    }

    /**
     * Generuje plik "urlset" na podstawie tablicy przekazanych urli
     * oraz o podanej nazwie, która jest przetwarzana na url. Zwraca fałsz,
     * jeśli niepowiedzie się generowanie pliku lub nazwę pliku w przypadku
     * prawidłowego wygenerowania pliku.
     *
     * @param array $urls
     * @param string $fileAffix
     * @return bool|string
     */
    public function generateUrlsetFile($urls, $fileAffix = '')
    {
        $arr = [
            'urlset' => [
                'xmlns' => self::XMLNS,
            ],
        ];

        if ($this->params['includePageImage'] || $this->params['includeNewsImage'] || $this->params['includeGalleryImages']) {
            $arr['urlset']['xmlns:image'] = self::XMLNS_IMAGE;
        }

        if (! empty($fileAffix)) {
            $fileAffix = '-' . $this->slugger->slugify($fileAffix);
        }

        $arr['urlset'] = array_merge($arr['urlset'], $urls);

        try {
            $filename = 'sitemap'.$fileAffix.'.xml';
            $array2xml = new Array2Xml();
            $xml = $array2xml->parse($arr);
            $xml->save($this->web_dir.$filename);
        } catch (\Exception $e) {
            return false;;
        }

        return $filename;
    }

    // -------------------------------------------------------------------------

    /**
     * Przygotowuje klucze "url" dla pliku typu "urlset".
     *
     * @param array $arr
     * @param bool $image Czy ma dodawać zdjęcia.
     * @return array
     */
    public function prepareUrls($arr, $image = false)
    {
        $out = [];

        foreach ($arr as $a) {
            $tmp = [
                'url' => [
                    'loc' => $a['canonical'],
                    'priority' => 0.5,
                ],
            ];

            if ($image && ! is_null($a['image'])) {
                $tmp['url']['image:image'] = [
                    'image:loc' => $a['image'],
                    'image:title' => html_entity_decode($a['title']),
                    'image:caption' => html_entity_decode($a['description']),
                ];
            }

            $out[] = $tmp;
        }

        return $out;
    }

    // -------------------------------------------------------------------------

    /**
     * Pobiera dane stron zdefiniowanych ręcznie.
     *
     * @return array
     */
    public function getManualPageSetArray()
    {
        return [
            $this->seo->getMetaArrayValues(),
        ];
    }

    /**
     * Pobiera aktualności dla generowania pliku sitemap.
     *
     * @return array
     */
    public function getNewsArray()
    {
        $news = $this->em->getRepository('WiAdminNewsBundle:News')->findForSitemap();
        $out = [];

        foreach ($news as $News) {
            if ($this->params['multifile'] && $this->params['splitNewsPerMonth']) {
                $date = ! is_null($News->getDateOfPublication()) ? $News->getDateOfPublication()->format('Y-m') : $News->getDateCreated()->format('Y-m');
                $out[$date][$News->getId()] = $this->seo->getMetaArrayValues($News);
            } else {
                $out[$News->getId()] = $this->seo->getMetaArrayValues($News);
            }
        }

        return $out;
    }

    public function getNewsCategoryArray()
    {
        $categories = $this->em->getRepository('WiAdminNewsBundle:Category')->findForSitemap();
        $out = [];

        foreach ($categories as $category) {
            if ($this->params['splitNewsPerMonth']) {
                $out[$category->getId()] = [
                    'category' => $category,
                    'dates' => [],
                ];

                foreach ($category->getNews() as $news) {
                    $date = ! is_null($news->getDateOfPublication()) ? $news->getDateOfPublication()->format('Y-m') : $news->getDateCreated()->format('Y-m');
                    $out[$category->getId()]['dates'][$date][$news->getId()] = $this->seo->getMetaArrayValues($news);
                }
            } else {
                $out[$category->getId()] = [
                    'category' => $category,
                    'news' => [],
                ];

                foreach ($category->getNews() as $news) {
                    $out[$category->getId()]['news'][$news->getId()] = $this->seo->getMetaArrayValues($news);
                }
            }
        }

        return $out;
    }

    /**
     * Pobiera strony dla generowania pliku sitemap.
     *
     * @return array
     */
    public function getPagesArray()
    {
        $pages = $this->em->getRepository('WiAdminPageBundle:Page')->findForSitemap(self::PAGE_IDS);
        $out = [];

        foreach ($pages as $page) {
            $out[$page->getId()] = $this->seo->getMetaArrayValues($page);
        }

        return $out;
    }

    /**
     * Pobiera strony z podziałem na grupy dla generowania pliku sitemap.
     *
     * @return array
     */
    public function getPageGroupsArray()
    {
        $groups = $this->em->getRepository('WiAdminPageBundle:PageGroup')->findForSitemap(self::PAGE_IDS);
        $out = [];

        foreach ($groups as $group) {
            $out[$group->getId()] = [
                'group' => $group->getName(),
                'pages' => [],
            ];

            foreach ($group->getPages() as $page) {
                $out[$group->getId()]['pages'][$page->getId()] = $this->seo->getMetaArrayValues($page);
            }
        }

        return $out;
    }
}
