<?php

namespace Wi\Admin\CoreBundle\Utils;

use GuzzleHttp\Client;

/**
 * Klasa pośrednicząca z Rest API.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class StrefaWiCMSAPI
{
    /**
     * Adres serwera API Strefy WiCMS.
     *
     * @const SERVER
     */
    const SERVER = 'http://strefawicms.webimpuls.pl/api/v1/';

    /**
     * Klucz API służący do autoryzacji w Strefie WiCMS.
     *
     * @const API_KEY
     */
    const API_KEY = 'ODIyYzJlODY5NjYyMGIzOTdjN2JjNTZmNGJiZWZk';

    /**
     * @var Client
     */
    private $client;

    /**
     * Konstruktor.
     *
     * @param array $config
     */
    public function __construct($config = [])
    {
        $this->init($config);
    }

    // -------------------------------------------------------------------------

    /**
     * Wysyła zapytanie POST dla sugestii.
     *
     * Wymagane parametry (niepuste):
     *      publicID - identyfikator instalacji,
     *      version - numer wersji WiCMS-a,
     *      administrator - login administratora
     *      currentUrl - adres, skąd wysłano sugestię,
     *      message - treść sugestii,
     *      firstName - imię osoby która wysłała sugestię,
     *
     * @param array $params
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function suggestionPost($params)
    {
        return $this->post('suggestions', $params);
    }

    /**
     * Wysyła zapytanie POST dla wiadomości do wsparcia technicznego.
     *
     * Wymagane parametry (niepuste):
     *      publicID - identyfikator instalacji,
     *      version - numer wersji WiCMS-a,
     *      administrator - login administratora
     *      currentUrl - adres, skąd wysłano sugestię,
     *      message - treść sugestii,
     *      firstName - imię osoby która wysłała sugestię,
     *
     * @param array $params
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function supportMessagePost($params)
    {
        return $this->post('supportMessages', $params);
    }

    /**
     * Wysyła zapytanie GET o identyfikator publiczny klienta.
     *
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function instanceGet()
    {
        return $this->get('instances');
    }

    /**
     * Wysyła zapytanie POST o utworzenie nowej instancji instalacji.
     *
     * Wymagane parametry (niepuste):
     *      publicID - identyfikator instalacji
     *      name - skrócona nazwa firmy
     *      server - adres IP servera
     *
     * @param array $params
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function instancePost($params)
    {
        return $this->post('instances', $params);
    }

    /**
     * Wysyła zapytanie POST o utworzenie zdarzenia logowania.
     *
     * Wymagane parametry (w body):
     *      publicID - identyfikator instalacji,
     *      administrator - login administratora
     *      server - adres IP serwera
     *      domain - nazwa domeny, na której pracuje instancja WiCMS,
     *
     * @param array $params
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function eventLoginPost($params)
    {
        try{
            $result = $this->post('eventLogin', $params);
        } catch (\Exception $e) {
            return false;
        }

        return $result;
    }

    /**
     * Wysyła zapytanie POST o utworzenie logu błędu.
     *
     * Wymagane parametry (w body):
     *      publicID - identyfikator instalacji,
     *      type - typ błędu (np. 500),
     *      message - treść błędu (cały stos),
     *      version - numer wersji WiCMS-a,
     *      administrator - login administratora (niewymagane)
     *
     * @param array $params
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function errorPost($params)
    {
        return $this->post('errors', $params);
    }

    /**
     * Wysyła zapytanie PUT o aktualizację logu błędu.
     *
     * Wymagane parametry (w body):
     *      publicID - identyfikator instalacji,
     *      comment - komentarz do błędu (niewymagane)
     *
     * @param int $id Id w adresie URI
     * @param array $params
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function errorPut($id, $params)
    {
        return $this->put('errors/'.$id, $params);
    }

    // -------------------------------------------------------------------------

    /**
     * Zapytanie GET do API.
     *
     * @param string $uri URI address
     * @param array $params Parametry GET
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function get($uri, $params = [])
    {
        if (! empty($params)) {
            $params = [
                'query' => $params,
            ];
        }

        try {
            $result = $this->client->get($uri, $params)->getBody()->getContents();
        } catch (\GuzzleHttp\Exception\ClientException $e) {
            return $e->getResponse()->getBody()->getContents();
        } catch (\Exception $e) {
            return false;
        }

        return $result;
    }

    /**
     * Zapytanie POST do API.
     *
     * @param string $uri URI address
     * @param array $params Parametry POST typu form_params
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function post($uri, $params = [])
    {
        if (! empty($params)) {
            $params = [
                'form_params' => $params,
            ];
        }

        try {
            $result = $this->client->post($uri, $params)->getBody()->getContents();
        } catch (\GuzzleHttp\Exception\ClientException $e) {
            return $e->getResponse()->getBody()->getContents();
        } catch (\Exception $e) {
            return false;
        }

        return $result;
    }

    /**
     * Zapytanie PUT do API.
     *
     * @param string $uri URI address
     * @param array $params Parametry PUT typu form_params
     * @return \GuzzleHttp\Psr7\Response|bool
     */
    public function put($uri, $params = [])
    {
        if (! empty($params)) {
            $params = [
                'form_params' => $params,
            ];
        }

        try {
            $result = $this->client->put($uri, $params)->getBody()->getContents();
        } catch (\GuzzleHttp\Exception\ClientException $e) {
            return $e->getResponse()->getBody()->getContents();
        } catch (\Exception $e) {
            return false;
        }

        return $result;
    }

    // -------------------------------------------------------------------------

    /**
     * Initializacja klienta cURL.
     *
     * @param array $config
     */
    private function init($config)
    {
        $api_key = array_key_exists('api_key', $config) ? $config['api_key'] : self::API_KEY;

        $this->client = new Client([
            'base_uri' => self::SERVER,
            'headers' => [
                'X-API-KEY' => $api_key,
            ],
        ]);
    }
}
