<?php

namespace Wi\Admin\FAQBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\FAQBundle\Entity\Question;
use Wi\Admin\FAQBundle\Form\QuestionDeleteType;
use Wi\Admin\FAQBundle\Form\QuestionType;

/**
 * Question controller.
 *
 * @author Maksymilian Baryczka <mmaxioo33@gmail.com>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class QuestionController extends Controller
{
    /**
     * Lists all question entities.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich aktywnych pytań.
        $paginator = $this->get('knp_paginator');
        $questions = $paginator->paginate(
            $em->getRepository('WiAdminFAQBundle:Question')
                ->getFindAllActiveQuery(
                    $request->query->getInt('category_id'),
                    $request->query->get('q')
                ),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 10
        );

        // Pobranie wszystkich kategorii.
        $categories = $em->getRepository('WiAdminFAQBundle:Category')->findAll();

        return $this->render('WiAdminFAQBundle:Question:index.html.php', [
            'categories' => $categories,
            'questions' => $questions,
        ]);
    }

    /**
     * Creates a new question entity.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie nowego obiektu pytań.
        $question = new Question();

        // Dodanie domyślnej kategorii, jeśli została przekazana oraz istnieje.
        if ($request->query->getInt('category_id')) {
            $category = $em->getRepository('WiAdminFAQBundle:Category')
                ->findOneById($request->query->getInt('category_id'))
            ;

            if (! is_null($category)) {
                $question->addCategory($category);
            }
        }

        // Ustawienie domyślnej daty publikacji.
        $question->setDateOfPublication(new \DateTime());

        // Generowanie formularza.
        $form = $this->createForm(QuestionType::class, $question);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Zapis.
            $em->persist($question);
            $em->flush();

            return $this->redirectToRoute('wi_admin_faq_question_show', [
                'id' => $question->getId(),
            ]);
        }

        return $this->render('WiAdminFAQBundle:Question:new.html.php', [
            'form' => $form->createView(),
            'question' => $question,
        ]);
    }

    /**
     * Show a question entity.
     *
     * @param Question $question
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Question $question)
    {
        return $this->render('WiAdminFAQBundle:Question:show.html.php', [
            'question' => $question,
        ]);
    }

    /**
     * Displays a form to edit an existing question entity.
     *
     * @param Request $request
     * @param Question $question
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Question $question)
    {
        // Generowanie formularza.
        $form = $this->createForm(QuestionType::class, $question);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($question);
            $em->flush();

            return $this->redirectToRoute('wi_admin_faq_question_show', [
                'id' => $question->getId(),
            ]);
        }

        return $this->render('WiAdminFAQBundle:Question:edit.html.php', [
            'form' => $form->createView(),
            'question' => $question,
        ]);
    }

    /**
     * Deletes a question entity.
     *
     * @param Request $request
     * @param Question $question
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, Question $question)
    {
        // Generowanie formularza.
        $deleteForm = $this->createForm(QuestionDeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usunięcie.
            $em->remove($question);
            $em->flush();

            return $this->redirectToRoute('wi_admin_faq_question_index');
        }

        return $this->render('WiAdminFAQBundle:Question:delete.html.php', [
            'deleteForm' => $deleteForm->createView(),
            'question' => $question,
        ]);
    }

    /**
     * Update a order category entity.
     *
     * @param Question $question
     * @param string $sort_dir
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function updateOrderAction(Question $question, $sort_dir)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $questionRepo = $em->getRepository('WiAdminFAQBundle:Question');

        // Pobranie wszystkich aktywnych pytań.
        $questions = $questionRepo->findAllActive();

        // Poprawienie kolejności. Może wystąpić sytuacja, że kilka kategorii
        // będzie miała przypisany ten sam numer w kolejności i wtedy
        // przenoszenie o '1' nie spełnia swojej roli.
        $fixOrder = function ($questions, $order = 1) use (&$fixOrder, $em) {
            foreach ($questions as $question) {
                $question->setOrderNr($order++);

                $em->persist($question);
            }

            $em->flush();
        };

        $fixOrder($questions);

        switch ($sort_dir) {
            case 'up': // Przesunięcie w górę.
                $nextTo = $questionRepo->findOneBy([
                    'orderNr' => $question->getOrderNr() - 1,
                ]);

                if (! is_null($nextTo)) {
                    $nextTo->setOrderNr($question->getOrderNr());
                    $question->setOrderNr($question->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            case 'down': // Przesunięcie w dół.
                $nextTo = $questionRepo->findOneBy([
                    'orderNr' => $question->getOrderNr() + 1,
                ]);

                if (! is_null($nextTo)) {
                    $question->setOrderNr($nextTo->getOrderNr());
                    $nextTo->setOrderNr($nextTo->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            default:
                break;
        }

        $em->persist($question);
        $em->flush();

        return $this->redirectToRoute('wi_admin_faq_question_index');
    }
}
