<?php

namespace Wi\Admin\LinkBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;

/**
 * Link
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 *
 * @ORM\Table(name="links")
 * @ORM\Entity(repositoryClass="Wi\Admin\LinkBundle\Repository\LinkRepository")
 * @ORM\EntityListeners({"Wi\Admin\LinkBundle\EventListener\Entity\LinkListener"})
 * @ORM\HasLifecycleCallbacks()
 * @Gedmo\SoftDeleteable(fieldName = "dateDeleted")
 * @UniqueEntity(fields="code", message="Code must be unique.")
 */
class Link
{
    const STATUS_ACTIVE = 'A';
    const STATUS_DELETED = 'D';
    const STATUS_DISABLED_REDIRECT = 'R';
    const STATUS_DISABLED_NOT_FOUND = 'N';

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="url", type="string", length=511)
     */
    private $url;

    /**
     * @var string
     *
     * @ORM\Column(name="code", type="string", length=64, unique=true)
     */
    private $code;

    /**
     * @var int
     *
     * @ORM\Column(name="count", type="integer")
     */
    private $count;

    /**
     * @var bool
     *
     * @ORM\Column(name="utm_enable", type="boolean")
     */
    private $utmEnable;

    /**
     * @var string
     *
     * @ORM\Column(name="utm_source", type="string", length=255, nullable=true)
     */
    private $utmSource;

    /**
     * @var string
     *
     * @ORM\Column(name="utm_medium", type="string", length=255, nullable=true)
     */
    private $utmMedium;

    /**
     * @var string
     *
     * @ORM\Column(name="utm_name", type="string", length=255, nullable=true)
     */
    private $utmName;

    /**
     * @var string
     *
     * @ORM\Column(name="utm_term", type="string", length=255, nullable=true)
     */
    private $utmTerm;

    /**
     * @var string
     *
     * @ORM\Column(name="utm_content", type="string", length=255, nullable=true)
     */
    private $utmContent;

    /**
     * @var string
     *
     * Dostępne statusy:
     *  A - aktywny (301)
     *  D - usunięty (404)
     *  R - wyłączony - przekierowanie na stronę główną (302)
     *  N - wyłączony - nie znaleziono (404)
     *
     * @ORM\Column(name="status", type="string", length=1, options={"default": "A", "fixed" = true})
     */
    private $status;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateCreated", type="datetime")
     */
    private $dateCreated;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateDeleted", type="datetime", nullable=true)
     */
    private $dateDeleted;


    // ------------------------------------------------------------
    // Has Lifecycle Callbacks methods.
    // ------------------------------------------------------------

    /**
     * Funkcja wykonuje się podczas wstawiania nowego rekordu do bazy.
     *
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        // Ustawienie daty utworzenia.
        $this->dateCreated = new \DateTime();

        // Domyślna wartość kliknięć.
        $this->count = 0;

        // Domyślny status.
        $this->status = self::STATUS_ACTIVE;

        // Generowanie kodu URL.
        $this->code = uniqid();
    }

    /**
     * Funkcja wykonuje się podczas aktualizacji rekordu bazy danych.
     *
     * @ORM\PreUpdate
     */
    public function preUpdate()
    {
        // Usunięcie tagów UTM, jeśli podczas edycji został wyłączony.
        if (! $this->utmEnable) {
            $this->utmSource = null;
            $this->utmMedium = null;
            $this->utmName = null;
            $this->utmTerm = null;
            $this->utmContent = null;
        }
    }

    // ------------------------------------------------------------
    // Getters and setters methods.
    // ------------------------------------------------------------

    /**
     * Set id
     *
     * @param int $id
     * @return Link
     */
    public function setId(int $id)
    {
        $this->id = $id;

        return $this;
    }

    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set url
     *
     * @param string $url
     * @return Link
     */
    public function setUrl($url)
    {
        $this->url = $url;

        return $this;
    }

    /**
     * Get url
     *
     * @return string
     */
    public function getUrl()
    {
        return $this->url;
    }

    /**
     * Set code
     *
     * @param string $code
     * @return Link
     */
    public function setCode($code)
    {
        $this->code = $code;

        return $this;
    }

    /**
     * Get code
     *
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * Set count
     *
     * @param integer $count
     * @return Link
     */
    public function setCount($count)
    {
        $this->count = $count;

        return $this;
    }

    /**
     * Get count
     *
     * @return int
     */
    public function getCount()
    {
        return $this->count;
    }

    /**
     * Set utmEnable
     *
     * @param boolean $utmEnable
     * @return Link
     */
    public function setUtmEnable($utmEnable)
    {
        $this->utmEnable = $utmEnable;

        return $this;
    }

    /**
     * Get utmEnable
     *
     * @return bool
     */
    public function getUtmEnable()
    {
        return $this->utmEnable;
    }

    /**
     * Set utmSource
     *
     * @param string $utmSource
     * @return Link
     */
    public function setUtmSource($utmSource)
    {
        $this->utmSource = $utmSource;

        return $this;
    }

    /**
     * Get utmSource
     *
     * @return string
     */
    public function getUtmSource()
    {
        return $this->utmSource;
    }

    /**
     * Set utmMedium
     *
     * @param string $utmMedium
     * @return Link
     */
    public function setUtmMedium($utmMedium)
    {
        $this->utmMedium = $utmMedium;

        return $this;
    }

    /**
     * Get utmMedium
     *
     * @return string
     */
    public function getUtmMedium()
    {
        return $this->utmMedium;
    }

    /**
     * Set utmName
     *
     * @param string $utmName
     * @return Link
     */
    public function setUtmName($utmName)
    {
        $this->utmName = $utmName;

        return $this;
    }

    /**
     * Get utmName
     *
     * @return string
     */
    public function getUtmName()
    {
        return $this->utmName;
    }

    /**
     * Set utmTerm
     *
     * @param string $utmTerm
     * @return Link
     */
    public function setUtmTerm($utmTerm)
    {
        $this->utmTerm = $utmTerm;

        return $this;
    }

    /**
     * Get utmTerm
     *
     * @return string
     */
    public function getUtmTerm()
    {
        return $this->utmTerm;
    }

    /**
     * Set utmContent
     *
     * @param string $utmContent
     * @return Link
     */
    public function setUtmContent($utmContent)
    {
        $this->utmContent = $utmContent;

        return $this;
    }

    /**
     * Get utmContent
     *
     * @return string
     */
    public function getUtmContent()
    {
        return $this->utmContent;
    }

    /**
     * Set status
     *
     * @param string $status
     * @return Link
     */
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set dateCreated
     *
     * @param \DateTime $dateCreated
     * @return Link
     */
    public function setDateCreated($dateCreated)
    {
        $this->dateCreated = $dateCreated;

        return $this;
    }

    /**
     * Get dateCreated
     *
     * @return \DateTime
     */
    public function getDateCreated()
    {
        return $this->dateCreated;
    }

    /**
     * Set dateDeleted
     *
     * @param \DateTime $dateDeleted
     * @return Link
     */
    public function setDateDeleted($dateDeleted)
    {
        $this->dateDeleted = $dateDeleted;

        return $this;
    }

    /**
     * Get dateDeleted
     *
     * @return \DateTime
     */
    public function getDateDeleted()
    {
        return $this->dateDeleted;
    }

    /**
     * Generuje adres URL z parametrami UTM jeśli sa.
     *
     * @return string
     */
    public function getUrlWithUtm()
    {
        $params = [];

        if ($this->getUtmEnable()) {
            if (! empty($source = $this->getUtmSource())) {
                $params['utm_source'] = $source;
            }

            if (! empty($medium = $this->getUtmMedium())) {
                $params['utm_medium'] = $medium;
            }

            if (! empty($name = $this->getUtmName())) {
                $params['utm_campaing'] = $name;
            }

            if (! empty($term = $this->getUtmTerm())) {
                $params['utm_term'] = $term;
            }

            if (! empty($content = $this->getUtmContent())) {
                $params['utm_content'] = $content;
            }
        }

        $parse = parse_url($this->url);

        if (isset($parse['query'])) {
            $parse['query'] .= (! empty($params) ? '&'.http_build_query($params) : null);
        } else {
            $parse['query'] = (! empty($params) ? http_build_query($params) : null);
        }

        return
            (isset($parse['scheme']) ? $parse['scheme'].'://' : '//').
            (isset($parse['user']) ? $parse['user'].(isset($parse['pass']) ? ':'.$parse['pass'] : '') . '@' : '').
            (isset($parse['host']) ? $parse['host'] : '').
            (isset($parse['port']) ? $parse['port'] : '').
            (isset($parse['path']) ? $parse['path'] : '').
            (isset($parse['query']) ? '?'.$parse['query'] : '').
            (isset($parse['fragment']) ? '#'.$parse['fragment'] : '')
        ;
    }
}
