<?php

namespace Wi\Admin\MenuBundle\Form;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Wi\Admin\MenuBundle\Entity\ItemMenu;

/**
 * Menu Item Delete Listener.
 *
 * @author Piotr Krzok <piotr.krzok@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class ItemMenuDeleteType extends AbstractType
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * Construct.
     *
     * @param EntityManagerInterface $em
     */
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add('deleteItemMenu', CheckboxType::class, [
                'attr' => [
                    'class' => 'switch-input',
                ],
                'mapped' => false,
                'required' => true,
            ])
        ;

        if ($options['data']->getChildren()->count()) {
            $builder
                ->add('deleteAll', CheckboxType::class, [
                    'attr' => [
                        'class' => 'switch-input',
                    ],
                    'mapped' => false,
                    'required' => false,
                ])
                ->add('newItemMenu', ChoiceType::class, [
                    'attr' => [
                        'class' => 'chosenSelectParent',
                    ],
                    'choice_attr' => function ($value, $key, $index) use ($options) {
                        if ($options['data']->getId() == $value->getId()) {
                            return ['disabled' => true];
                        }

                        $children = [];

                        $setDisabled = function ($itemMenus) use (&$setDisabled, &$children) {
                            foreach ($itemMenus as $itemMenu) {
                                $children[] = $itemMenu->getId();

                                if ($itemMenu->getChildren()->count()) {
                                    $setDisabled($itemMenu->getChildren());
                                }
                            }
                        };

                        $setDisabled($options['data']->getChildren());

                        if (in_array($value->getId(), $children)) {
                            return ['disabled' => true];
                        }

                        return [];
                    },
                    'choice_label' => function ($value, $key, $index) {
                        return $value->label;
                    },
                    'choice_translation_domain' => false,
                    'choices' => $this->getChoices(),
                    'mapped' => false,
                    'placeholder' => 'menuDelete.newItem',
                    'required' => false,
                ])
            ;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(array(
            'data_class' => ItemMenu::class,
        ));
    }

    /**
     * {@inheritdoc}
     */
    public function getBlockPrefix()
    {
        return '';
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Przetwarza pozycje menu i generuje z nich drzewo.
     *
     * @return ItemMenu[]
     */
    private function getChoices()
    {

        $choices = [];

        // Pobranie wszystkich głównych, aktywnych grup stron.
        $menuItems = $this->em->getRepository('WiAdminMenuBundle:ItemMenu')
            ->findAllActiveWithoutParent()
        ;

        /**
         * Przetwarza drzewo zagnieżdżeń menu i wypisuje do selecta.
         *
         * @param   ItemMenu[] $menuItems
         * @param   string      $prefix
         */
        $createTree = function ($menuItems, $prefix = '') use (&$createTree, &$choices) {
            foreach ($menuItems as $menuItem) {
                // Pole label jest tworzone dynamicznie tylko na potrzeby wypisania etykiety.
                $menuItem->label = $prefix . $menuItem->getName() . ' - ' . $menuItem->getLocation()->getName();
                $choices[] = $menuItem;

                if ($menuItem->getChildren()->count()) {
                    $createTree($menuItem->getChildren(), $prefix . '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;');
                }
            }

        };

        $createTree($menuItems);

        return $choices;
    }
}
