<?php

namespace Wi\Admin\NewsBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\NewsBundle\Entity\Category;
use Wi\Admin\NewsBundle\Form\CategoryDeleteType;
use Wi\Admin\NewsBundle\Form\CategoryType;

/**
 * Category controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class CategoryController extends Controller
{

    /**
     * Lists all category entities.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request)
    {
        $route = $request->attributes->get('_route');
        $type = Category::TYPE_NEWS;
        if ($route == 'wi_admin_blog_categories_index') {
            $type = Category::TYPE_BLOG;
        }

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich aktywnych kategorii.
        $categories = $em->getRepository('WiAdminNewsBundle:Category')
            ->findAllActiveWithoutParent($type)
        ;

        if ($route == 'wi_admin_blog_categories_index') {
            return $this->render('WiAdminNewsBundle:BlogCategory:index.html.php', [
                'categories' => $categories,
            ]);
        }

        return $this->render('WiAdminNewsBundle:Category:index.html.php', [
            'categories' => $categories,
        ]);
    }

    /**
     * Creates a new category entity.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        $route = $request->attributes->get('_route');
        $type = Category::TYPE_NEWS;
        if ($route == 'wi_admin_blog_categories_new') {
            $type = Category::TYPE_BLOG;
        }

        // Utworzenie nowego obiektu kategorii.
        $category = new Category();

        // Przypisanie typu kategorii.
        $category->setType($type);

        // Generowanie formularza.
        $form = $this->createForm(CategoryType::class, $category, ['type' => $type]);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($category);
            $em->flush();

            if ($route == 'wi_admin_blog_categories_new') {
                return $this->redirectToRoute('wi_admin_blog_categories_show', [
                    'id' => $category->getId(),
                ]);
            }
            return $this->redirectToRoute('wi_admin_news_categories_show', [
                'id' => $category->getId(),
            ]);
        }
        if ($route == 'wi_admin_blog_categories_new') {
            return $this->render('WiAdminNewsBundle:BlogCategory:new.html.php', [
                'category' => $category,
                'form' => $form->createView(),
            ]);
        }
        return $this->render('WiAdminNewsBundle:Category:new.html.php', [
            'category' => $category,
            'form' => $form->createView(),
        ]);
    }

    /**
     * Finds and displays a category entity.
     *
     * @param   Category $category
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Category $category, Request $request)
    {
        $route = $request->attributes->get('_route');
        if ($route == 'wi_admin_blog_categories_show') {
            return $this->render('WiAdminNewsBundle:BlogCategory:show.html.php', [
                'category' => $category,
            ]);
        }
        return $this->render('WiAdminNewsBundle:Category:show.html.php', [
            'category' => $category,
        ]);
    }

    /**
     * Displays a form to edit an existing category entity.
     *
     * @param   Request $request
     * @param   Category $category
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Category $category)
    {
        $route = $request->attributes->get('_route');
        $type = Category::TYPE_NEWS;
        if ($route == 'wi_admin_blog_categories_edit') {
            $type = Category::TYPE_BLOG;
        }

        // Generowanie formularza.
        $editForm = $this->createForm(CategoryType::class, $category, ['type' => $type]);
        $editForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($editForm->isSubmitted() && $editForm->isValid()) {
            // Zapis.
            $this->getDoctrine()->getManager()->flush();

            if ($route == 'wi_admin_blog_categories_edit') {
                return $this->redirectToRoute('wi_admin_blog_categories_show', [
                    'id' => $category->getId(),
                ]);
            }
            return $this->redirectToRoute('wi_admin_news_categories_show', [
                'id' => $category->getId(),
            ]);
        }

        if ($route == 'wi_admin_blog_categories_edit') {
            return $this->render('WiAdminNewsBundle:BlogCategory:edit.html.php', [
                'category' => $category,
                'edit_form' => $editForm->createView(),
            ]);
        }
        return $this->render('WiAdminNewsBundle:Category:edit.html.php', [
            'category' => $category,
            'edit_form' => $editForm->createView(),
        ]);
    }

    /**
     * Deletes a category entity.
     *
     * @param   Request $request
     * @param   Category $category
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, Category $category)
    {
        $route = $request->attributes->get('_route');
        $type = Category::TYPE_NEWS;
        if ($route == 'wi_admin_blog_categories_delete') {
            $type = Category::TYPE_BLOG;
        }

        // Utworzenie formularza usuwania kategorii.
        $formDelete = $this->createForm(CategoryDeleteType::class, $category, ['type' => $type]);
        $formDelete->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($formDelete->isSubmitted() && $formDelete->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            if ($category->getNews()->count() || $category->getChildren()->count()) {
                // Usuwanie wszystkich wpisów i kategorii.
                if ($formDelete->get('deleteAll')->getData()) {
                    // Rekursywne usuwanie kategorii.
                    $recursiveRemove = function ($category) use (&$recursiveRemove, $em) {
                        if ($category->getChildren()->count() > 0) {
                            foreach ($category->getChildren() as $child) {
                                $recursiveRemove($child);
                            }
                        }

                        // Usuwanie wpisów.
                        foreach ($category->getNews() as $news) {
                            $em->remove($news);
                            $em->flush();
                        }

                        $em->remove($category);
                    };

                    $recursiveRemove($category);
                } else {
                    // Usuwanie kategorii i odpięcie newsów.
                    if ($formDelete->get('noDeleteNews')->getData() == 'unpin') {
                        foreach ($category->getNews() as $news) {
                            // Odpięcie kategorii.
                            $news->removeCategory($category);

                            if ($news->getCategories()->count() == 0) {
                                // Usunięcie aktualności, jeśli nie ma innej kategorii.
                                $em->remove($news);
                            } else {
                                // Przygotowanie do zapisu.
                                $em->persist($news);
                            }
                        }

                        foreach ($category->getChildren() as $child) {
                            // Usunięcie rodzica podkategorii.
                            $child->setParent(null);

                            // Przygotowanie do zapisu.
                            $em->persist($child);
                        }

                        // Zapis edytowanych aktualności i podkategorii.
                        $em->flush();
                    } else {
                        // Usunięcie kategorii i przenoszenie aktualności i podkategorii.
                        $newCategory = $formDelete->get('newCategory')->getData();

                        foreach ($category->getNews() as $news) {
                            // Usunięcie kategorii.
                            $news->removeCategory($category);

                            // Ustawienie nowej kategorii, jeśli nie była przypisana.
                            if (! $news->getCategories()->contains($newCategory)) {
                                // Dodanie nowej kategorii.
                                $news->addCategory($newCategory);
                            }

                            // Przygotowanie do zapisu.
                            $em->persist($news);
                        }

                        // Przypisanie nowego rodzica podkategoriom.
                        foreach ($category->getChildren() as $child) {
                            // Ustawienie nowego rodzica.
                            $child->setParent($newCategory);

                            // Przygotowanie do zapisu.
                            $em->persist($child);
                        }

                        // Zapis edytowanych aktualności i podkategorii.
                        $em->flush();
                    }
                }
            }

            // Usuwanie kategorii.
            $em->remove($category);
            $em->flush();

            // Ustawienie daty usunięcia w kategorii ze statusem 'D'. Przy usuwaniu
            // jest problem ustawić dlatego ten kod się pojawił.
            $deletedCategories = $em->getRepository('WiAdminNewsBundle:Category')
                ->findBy([
                    'dateDeleted' => null,
                    'status' => 'D',
                ])
            ;

            if (count($deletedCategories) > 0) {
                foreach ($deletedCategories as $deletedCategory) {
                    $deletedCategory->setDateDeleted(new \DateTime());
                    $em->persist($deletedCategory);
                }

                $em->flush();
            }

            if ($route == 'wi_admin_blog_categories_delete') {
                return $this->redirectToRoute('wi_admin_blog_categories_index');
            }
            return $this->redirectToRoute('wi_admin_news_categories_index');
        }

        if ($route == 'wi_admin_blog_categories_delete') {
            return $this->render('WiAdminNewsBundle:BlogCategory:delete.html.php', [
                'category' => $category,
                'formDelete' => $formDelete->createView(),
            ]);
        }
        return $this->render('WiAdminNewsBundle:Category:delete.html.php', [
            'category' => $category,
            'formDelete' => $formDelete->createView(),
        ]);
    }

    /**
     * Update a order category entity.
     *
     * @param Category $category
     * @param string $sort_dir
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function updateOrderAction(Category $category, $sort_dir, Request $request)
    {
        $route = $request->attributes->get('_route');
        $type = Category::TYPE_NEWS;
        if ($route == 'wi_admin_blog_categories_update_order') {
            $type = Category::TYPE_BLOG;
        }

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $categoryRepo = $em->getRepository('WiAdminNewsBundle:Category');

        // Pobranie wszystkich aktywnych kategorii.
        $categories = $categoryRepo->findAllActiveWithoutParent($type);

        // Poprawienie kolejności. Może wystąpić sytuacja, że kilka kategorii
        // będzie miała przypisany ten sam numer w kolejności i wtedy
        // przenoszenie o '1' nie spełnia swojej roli.
        $fixOrder = function ($categories, $order = 1) use (&$fixOrder, $em) {
            foreach ($categories as $category) {
                $category->setOrderNr($order++);

                $em->persist($category);

                if ($category->getChildren()->count()) {
                    $fixOrder($category->getChildren(), 1);
                }
            }

            $em->flush();
        };

        $fixOrder($categories);

        switch ($sort_dir) {
            case 'up': // Przesunięcie w górę.
                $nextTo = $categoryRepo->findOneBy([
                    'orderNr' => $category->getOrderNr() - 1,
                    'parent' => $category->getParent(),
                ]);

                if (! is_null($nextTo)) {
                    $nextTo->setOrderNr($category->getOrderNr());
                    $category->setOrderNr($category->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            case 'down': // Przesunięcie w dół.
                $nextTo = $categoryRepo->findOneBy([
                    'orderNr' => $category->getOrderNr() + 1,
                    'parent' => $category->getParent(),
                ]);

                if (! is_null($nextTo)) {
                    $category->setOrderNr($nextTo->getOrderNr());
                    $nextTo->setOrderNr($nextTo->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            default:
                break;
        }

        $em->persist($category);
        $em->flush();

        if ($route == 'wi_admin_blog_categories_update_order') {
            return $this->redirectToRoute('wi_admin_blog_categories_index');
        }
        return $this->redirectToRoute('wi_admin_news_categories_index');
    }
}
