<?php

namespace Wi\Admin\NewsBundle\Form;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Wi\Admin\NewsBundle\Entity\Category;

/**
 * Category Delete Type.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class CategoryDeleteType extends AbstractType
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * Construct.
     *
     * @param EntityManagerInterface $em
     */
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add('deleteCategory', CheckboxType::class, [
                'attr' => [
                    'class' => 'switch-input',
                ],
                'mapped' => false,
                'required' => true,
            ])
        ;

        if ($options['data']->getNews()->count() || $options['data']->getChildren()->count()) {
            $builder
                ->add('deleteAll', CheckboxType::class, [
                    'attr' => [
                        'class' => 'switch-input',
                    ],
                    'mapped' => false,
                    'required' => false,
                ])
                ->add('noDeleteNews', ChoiceType::class, [
                    'choices' => [
                        'catDelete.unpin' => 'unpin',
                        'catDelete.changeCat' => 'changeCat',
                    ],
                    'expanded' => true,
                    'mapped' => false,
                    'multiple' => false,
                ])
                ->add('newCategory', ChoiceType::class, [
                    'choice_attr' => function ($value, $key, $index) use ($options) {
                        if ($options['data']->getId() == $value->getId()) {
                            return ['disabled' => true];
                        }

                        $children = [];

                        $setDisabled = function ($categories) use (&$setDisabled, &$children) {
                            foreach ($categories as $category) {
                                $children[] = $category->getId();

                                if ($category->getChildren()->count()) {
                                    $setDisabled($category->getChildren());
                                }
                            }
                        };

                        $setDisabled($options['data']->getChildren());

                        if (in_array($value->getId(), $children)) {
                            return ['disabled' => true];
                        }

                        return [];
                    },
                    'choice_label' => function ($value, $key, $index) {
                        return $value->label;
                    },
                    'choice_translation_domain' => false,
                    'choices' => $this->getChoices($options['type']),
                    'mapped' => false,
                    'placeholder' => 'groupDelete.newGroup',
                    'required' => false,
                ])
            ;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(array(
            'data_class' => Category::class,
            'type' => null,
        ));
    }

    /**
     * {@inheritdoc}
     */
    public function getBlockPrefix()
    {
        return '';
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Przetwarza kategorie i generuje z nich drzewo.
     *
     * @return Category[]
     */
    private function getChoices($type)
    {
        $choices = [];

        // Pobranie wszystkich głównych, aktywnych kategorii.
        $categories = $this->em->getRepository('WiAdminNewsBundle:Category')
            ->findAllActiveWithoutParent($type)
        ;

        /**
         * Przetwarza drzewo zagnieżdżeń kategorii i wypisuje do selecta.
         *
         * @param   Category[]  $categories
         * @param   string      $prefix
         */
        $createTree = function ($categories, $prefix = ' ') use (&$createTree, &$choices) {
            foreach ($categories as $category) {
                // Pole label jest tworzone dynamicznie tylko na potrzeby wypisania etykiety.
                $category->label = $prefix . $category->getName();

                $choices[] = $category;

                if ($category->getChildren()->count()) {
                    $createTree($category->getChildren(), '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;' . $prefix);
                }
            }

        };

        $createTree($categories);

        return $choices;
    }
}
