<?php

namespace Wi\Admin\NewsBundle\Form;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Wi\Admin\NewsBundle\Entity\Category;

/**
 * Category Type.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class CategoryType extends AbstractType
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * Construct.
     *
     * @param EntityManagerInterface $em
     */
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add('name', TextType::class, [
                'attr' => [
                    'maxlength' => 45,
                    'placeholder' => 'categoryForm.name',
                ],
                'label' => 'categoryForm.name',
            ])
            ->add('orderNr', null, [
                'attr' => [
                    'max' => 2147483646,
                    'maxlength' => 11,
                    'min' => -2147483646,
                ],
                'label' => 'categoryForm.orderNr',
            ])
            ->add('description', TextareaType::class, [
                'attr' => [
                    'maxlength' => 65535,
                    'placeholder' => 'categoryForm.description',
                ],
                'label' => 'categoryForm.description',
                'required' => false,
            ])
            ->add('parent', null, [
                'attr' => [
                    'class' => 'chosenSelectCategory',
                ],
                'choices' => $this->getChoices($options['type']),
                'choice_label' => function ($value, $key, $index) {
                    return $value->label;
                },
                'choice_attr' => function ($value, $key, $index) use ($options) {
                    if ($options['data']->getId() == $value->getId()) {
                        return ['disabled' => true];
                    }
                    return [];
                },
                'label' => 'categoryForm.parent',
                'label_attr' => [
                    'class' => ''
                ],
                'required' => false,
                'placeholder' => 'categoryForm.parentPlace',
            ])
        ;
    }

    /**
     * {@inheritdoc}
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(array(
            'data_class' => Category::class,
            'type' => null,
        ));
    }

    /**
     * {@inheritdoc}
     */
    public function getBlockPrefix()
    {
        return 'wi_admin_newsbundle_category';
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Przetwarza kategorie i generuje z nich drzewo.
     *
     * @return Category[]
     */
    private function getChoices($type)
    {
        $choices = [];

        // Pobranie wszystkich głównych, aktywnych grup stron.
        $categories = $this->em->getRepository('WiAdminNewsBundle:Category')
            ->findAllActiveWithoutParent($type)
        ;

        /**
         * Przetwarza drzewo zagnieżdżeń kategorii i wypisuje do selecta.
         *
         * @param   Category[]  $categories
         * @param   string      $prefix
         */
        $createTree = function ($categories, $prefix = ' ') use (&$createTree, &$choices) {
            foreach ($categories as $category) {
                // Pole label jest tworzone dynamicznie tylko na potrzeby wypisania etykiety.
                $category->label = $prefix . $category->getName();

                $choices[] = $category;

                if ($category->getChildren()->count()) {
                    $createTree($category->getChildren(), '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;' . $prefix);
                }
            }

        };

        $createTree($categories);

        return $choices;
    }
}
