<?php

namespace Wi\Admin\NewsBundle\Form;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\DateTimeType;
use Symfony\Component\Form\Extension\Core\Type\FileType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Wi\Admin\NewsBundle\Entity\Category;
use Wi\Admin\NewsBundle\Entity\News;
use Wi\Admin\NewsBundle\Entity\Tag;

/**
 * News Type.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class NewsType extends AbstractType
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * Construct.
     *
     * @param EntityManagerInterface $em
     */
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add('movie', TextType::class, [
                'attr' => [
                    'maxlength' => 255,
                    'placeholder' => 'newsNew.form.movieTitle',
                ],
                'required' => false,
            ])
            ->add('title', TextType::class, [
                'attr' => [
                    'maxlength' => 256,
                    'placeholder' => 'newsNew.form.titleTitle',
                ],
            ])
            ->add('shortcut', TextareaType::class, [
                'attr' => [
                    'maxlength' => 512,
                    'placeholder' => 'newsNew.form.shortTitle',
                ],
                'required' => false,
            ])
            ->add('content', TextareaType::class, [
                'attr' => [
                    'maxlength' => 65535,
                    'rows' => 10,
                    'class' => 'tinyMCE',
                ],
                'required' => false,
            ])
            ->add('dateOfPublication', DateTimeType::class, [
                'attr' => [
                    'class' => 'datePicker',
                ],
                'format' => 'dd-MM-yyyy',
                'widget' => 'single_text',
                'required' => false,
            ])
            ->add('metaTitle', TextType::class, [
                'attr' => [
                    'maxlength' => 256,
                    'placeholder' => 'newsNew.form.placeMetaTitle',
                ],
                'required' => false,
            ])
            ->add('metaDescription', TextareaType::class, [
                'attr' => [
                    'maxlength' => 512,
                    'placeholder' => 'newsNew.form.placeMetaDescription',
                ],
                'required' => false,
            ])
            ->add('metaKeywords', TextareaType::class, [
                'attr' => [
                    'maxlength' => 256,
                    'placeholder' => 'newsNew.form.placeMetaKeywords',
                ],
                'required' => false,
            ])
            ->add('metaCanonical', TextType::class, [
                'attr' => [
                    'maxlength' => 512,
                    'placeholder' => 'newsNew.form.placeMetaCanonical',
                ],
                'required' => false,
            ])
            ->add('metaImage', TextType::class, [
                'attr' => [
                    'maxlength' => 256,
                    'placeholder' => 'newsNew.form.placeMetaImage',
                ],
                'required' => false,
            ])
            ->add('categories', null, [
                'attr' => [
                    'class' => 'chosenSelectCategory',
                ],
                'choices' => $this->getChoices($options['type']),
                'choice_label' => function ($value, $key, $index) {
                    return $value->label;
                },
                'required' => false,
            ])
            ->add('status', ChoiceType::class, [
                'choices' => [
                    'Activ' => 'A',
                    'Unpublished' => 'N',
                ],
                'attr' => [
                    'class' => 'chosenSelectCategory',
                ],
                'label' => 'marketAucNew.form.status',
                'required' => true,
            ])
            ->add('tags', EntityType::class, [
                'attr' => [
                    'class' => 'tagsSelect',
                    'placeholder' => 'newsNew.form.tagsPlace',
                ],
                'label' => 'newsNew.form.tagsLabel',
                'class' => Tag::class,
                'choice_label' => 'name',
                'multiple' => true,
                'required' => false,
            ])
            ->add('enableDownload', CheckboxType::class, [
                'attr' => [
                    'class' => 'switch-input',
                ],
                'label' => false,
                'required' => false,
            ])
        ;
    }

    /**
     * {@inheritdoc}
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(array(
            'data_class' => News::class,
            'type' => null,
        ));
    }

    /**
     * {@inheritdoc}
     */
    public function getBlockPrefix()
    {
        return 'wi_admin_newsbundle_news';
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Przetwarza kategorie i generuje z nich drzewo.
     *
     * @return Category[]
     */
    private function getChoices($type)
    {
        $choices = [];

        // Pobranie wszystkich głównych, aktywnych grup stron.
        $categories = $this->em->getRepository('WiAdminNewsBundle:Category')
            ->findAllActiveWithoutParent($type)
        ;

        /**
         * Przetwarza drzewo zagnieżdżeń kategorii i wypisuje do selecta.
         *
         * @param   Category[]  $categories
         * @param   string      $prefix
         */
        $createTree = function ($categories, $prefix = ' ') use (&$createTree, &$choices) {
            foreach ($categories as $category) {
                // Pole label jest tworzone dynamicznie tylko na potrzeby wypisania etykiety.
                $category->label = $prefix . $category->getName();

                $choices[] = $category;

                if ($category->getChildren()->count()) {
                    $createTree($category->getChildren(), '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;' . $prefix);
                }
            }

        };

        $createTree($categories);

        return $choices;
    }
}
