<?php

namespace Wi\Admin\NewsBundle\Repository;

use Doctrine\ORM\EntityRepository;
use Wi\Admin\NewsBundle\Entity\News;
use Wi\Admin\UserBundle\Entity\User;

/**
 * NewsRepository
 *
 * This class was generated by the Doctrine ORM. Add your own custom
 * repository methods below.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class NewsRepository extends EntityRepository
{
    /**
     * Liczba wszystkich aktywnych aktualności.
     *
     * @return int
     */
    public function countActive($type)
    {
        $qb = $this->createQueryBuilder('n');

        return (int) $qb
            ->select($qb->expr()->count('n'))
            ->where('n.status = :status')
            ->andWhere('n.type = :type')
            ->setParameters([
                'status' => 'A',
                'type' => $type,
            ])
            ->getQuery()
            ->getSingleScalarResult()
        ;
    }

    /**
     * Get news with limit by category.
     *
     * @param   int     $category_id  News from category. When is null, from
     *                                all categories.
     * @param   int     $limit        Limit of news.
     * @return  array
     */
    public function findByCategoryWithLimit($type, $category_id = null, $limit = 3)
    {
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('n')
            ->addSelect('c, i')
            ->leftJoin('n.categories', 'c')
            ->leftJoin('n.images', 'i')
            ->where('n.status = :status')
            ->andWhere('n.dateOfPublication <= :dateOfPublication')
            ->andWhere('n.dateDeleted IS NULL')
            ->andWhere('n.type = :type')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
            'dateOfPublication' => new \DateTime(),
            'type' => $type,
        ];

        // Jeśli ID kategorii zostało przekazane to zawęźić do tej kategorii.
        if ($category_id) {
            $params['category_id'] = $category_id;
            $qb->andWhere('c.id = :category_id');
        }

        return $qb
            ->orderBy('n.dateOfPublication', 'DESC')
            ->setMaxResults($limit)
            ->setParameters($params)
            ->getQuery()
            ->getResult();
        ;
    }

    /**
     * Get news with not limit by category.
     *
     * @param   int     $category_id  News from category. When is null, from
     *                                all categories.
     * @return  array
     */
    public function findByCategoryNotLimit($category_id = null)
    {
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('n')
            ->addSelect('c, i')
            ->leftJoin('n.categories', 'c')
            ->leftJoin('n.images', 'i')
            ->where('n.status = :status')
            ->andWhere('n.dateOfPublication <= :dateOfPublication')
            ->andWhere('n.dateDeleted IS NULL')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
            'dateOfPublication' => new \DateTime(),
        ];

        // Jeśli ID kategorii zostało przekazane to zawęźić do tej kategorii.
        if ($category_id) {
            $params['category_id'] = $category_id;
            $qb->andWhere('c.id = :category_id');
        }

        return $qb
            ->orderBy('n.dateOfPublication', 'DESC')
            ->setParameters($params)
            ->getQuery()
            ->getResult();
        ;
    }

    /**
     * Pobiera zapytanie dla wszystkich aktywnych aktualności.
     *
     * @param   int     $category_id    ID kategorii do filtrowania aktualności.
     *                                  Domyślnie 'null'.
     * @return  Query
     */
    public function getFindAllActiveQuery($type, $category_id = null, $search = '')
    {
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('n')
            ->addSelect('c, i')
            ->leftJoin('n.categories', 'c')
            ->leftJoin('n.images', 'i')
            ->where('n.status = :status')
            ->orWhere('n.status = :statusN')
            ->andWhere('n.dateDeleted IS NULL')
            ->andWhere('n.type = :type')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
            'statusN' => 'N',
            'type' => $type,
        ];

        // Jeśli ID kategorii zostało przekazane to zawęźić do tej kategorii.
        if ($category_id) {
            $params['category_id'] = $category_id;
            $qb->andWhere('c.id = :category_id');
        }

        // Jeśli tekst do wyszukania jest przekazany to szukaj.
        if ($search) {
            $params['search'] = '%' . $search . '%';
            $qb->andWhere($qb->expr()->orX(
                $qb->expr()->like('n.title', ':search'),
                $qb->expr()->like('n.shortcut', ':search'),
                $qb->expr()->like('n.content', ':search'),
                $qb->expr()->like('n.metaTitle', ':search'),
                $qb->expr()->like('n.metaDescription', ':search'),
                $qb->expr()->like('n.metaKeywords', ':search'),
                $qb->expr()->like('n.metaCanonical', ':search'),
                $qb->expr()->like('n.metaImage', ':search'),
                $qb->expr()->like('n.visitCounter', ':search')
            ));
        }

        return $qb
            ->setParameters($params)
            ->getQuery()
        ;
    }

    /**
     * Pobiera zapytanie dla wszystkich aktywnych i opublikowanych aktualności.
     *
     * @param   int     $category_id    ID kategorii do filtrowania aktualności.
     *                                  Domyślnie 'null'.
     * @param   string  $search         Wyszukiwana fraza.
     *
     * @return  Query
     */
    public function getFindAllActivePublishedQuery($type, $category_id = null, $search='')
    {
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('n')
            ->addSelect('c, i')
            ->leftJoin('n.categories', 'c')
            ->leftJoin('n.images', 'i')
            ->where('n.status = :status')
            ->andWhere('n.dateDeleted IS NULL')
            ->andWhere('n.dateOfPublication <= :dateOfPublication')
            ->andWhere('n.type = :type')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
            'dateOfPublication' => new \DateTime(),
            'type' => $type,
        ];

        // Jeśli ID kategorii zostało przekazane to zawęźić do tej kategorii.
        if ($category_id) {
            $params['category_id'] = $category_id;
            $qb->andWhere('c.id = :category_id');
        }

        // Jeśli tekst do wyszukania jest przekazany to szukaj.
        if ($search) {
            $params['search'] = '%' . $search . '%';
            $qb->andWhere($qb->expr()->orX(
                $qb->expr()->like('n.title', ':search'),
                $qb->expr()->like('n.shortcut', ':search'),
                $qb->expr()->like('n.content', ':search'),
                $qb->expr()->like('n.metaTitle', ':search'),
                $qb->expr()->like('n.metaDescription', ':search'),
                $qb->expr()->like('n.metaKeywords', ':search'),
                $qb->expr()->like('n.metaCanonical', ':search'),
                $qb->expr()->like('n.metaImage', ':search'),
                $qb->expr()->like('n.visitCounter', ':search')
            ));
        }

        return $qb
            ->orderBy('n.dateOfPublication', 'DESC')
            ->setParameters($params)
            ->getQuery()
        ;
    }

    /**
     * Wyszukiwanie po tytule.
     *
     * @param string
     * @return array
     */

    public function findByTitleWithSerach($search, $type){
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('n')
            ->where('n.status = :status')
            ->orWhere('n.status = :statusN')
            ->andWhere('n.dateDeleted IS NULL')
            ->andWhere('n.type = :type')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
            'statusN' => 'N',
            'type' => $type,
        ];

        if ($search) {
            $params['search'] = '%' . $search . '%';
            $qb->andWhere($qb->expr()->orX(
                $qb->expr()->like('n.title', ':search')
            ));
        }

        return $qb
            ->orderBy('n.title', 'DESC')
            ->setParameters($params)
            ->getQuery()
            ->getResult();
        ;
    }

    /**
     * Wyszukiwanie po tytule po stronie Frontu.
     *
     * @param string
     * @return array
     */

    public function findByTitleWithSerachPublic($search, $type){
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('n')
            ->where('n.status = :status')
            ->andWhere('n.dateDeleted IS NULL')
            ->andWhere('n.type = :type')
            ->andWhere('n.dateOfPublication <= :dateOfPublication')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
            'type' => $type,
            'dateOfPublication' => new \DateTime(),
        ];

        if ($search) {
            $params['search'] = '%' . $search . '%';
            $qb->andWhere($qb->expr()->orX(
                $qb->expr()->like('n.title', ':search')
            ));
        }

        return $qb
            ->orderBy('n.title', 'DESC')
            ->setParameters($params)
            ->getQuery()
            ->getResult();
        ;
    }

    // ------------------------------------------------------------
    // API methods.
    // ------------------------------------------------------------

    /**
     * Liczba wszystkich wpisów. Jeśli przekazane ID kategorii to liczba
     * wszystkich wpisów w danej kategorii.
     *
     * @param int $category_id
     * @return int
     */
    public function countActivePublished($category_id = null)
    {
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('n');
        $qb
            ->select($qb->expr()->countDistinct('n'))
            ->leftJoin('n.categories', 'c')
            ->where('n.status = :status')
            ->andWhere('n.dateDeleted IS NULL')
            ->andWhere('n.dateOfPublication <= :dateOfPublication')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
            'dateOfPublication' => new \DateTime(),
        ];

        // Jeśli ID kategorii zostało przekazane to pobrać tylko z tej kategorii.
        if ($category_id) {
            $params['category_id'] = (int) $category_id;
            $qb->andWhere('c.id = :category_id');
        }

        return (int) $qb
            ->setParameters($params)
            ->getQuery()
            ->getSingleScalarResult()
        ;
    }

    /**
     * Pobiera wszystkie wpisy w zależności od parametrów. Jeśli przekazane jest
     * ID kategorii to pobiera wszystkie z danej kategorii. Jeśli jest przekazany
     * numer strony oraz liczba wyników na stronę to wyniki są stronicowane.
     * Sortowanie może być malejące (desc) lub rosnące (asc).
     *
     * @param int $category_id
     * @param int $page
     * @param int $per_page
     * @param string $sort_dir
     * @return array
     */
    public function findActivePublished($category_id = null, $page = null, $per_page = null, $sort_dir = 'desc')
    {
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('n')
            ->leftJoin('n.categories', 'c')
            ->where('n.status = :status')
            ->andWhere('n.dateDeleted IS NULL')
            ->andWhere('n.dateOfPublication <= :dateOfPublication')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
            'dateOfPublication' => new \DateTime(),
        ];

        // Jeśli ID kategorii zostało przekazane to pobrać tylko z tej kategorii.
        if ($category_id) {
            $params['category_id'] = (int) $category_id;
            $qb->andWhere('c.id = :category_id');
        }

        if (! empty($page) && ! empty($per_page)) {
            $limit = (int) $per_page;
            $offset = ($page - 1) * $per_page;

            $qb
                ->setFirstResult($offset)
                ->setMaxResults($limit)
            ;
        }

        return $qb
            ->setParameters($params)
            ->orderBy('n.dateOfPublication', $sort_dir)
            ->distinct()
            ->getQuery()
            ->getResult()
        ;
    }

    // -------------------------------------------------------------------------

    /**
     * Pobiera aktualności dla generowania pliku sitemap.
     *
     * @return array
     */
    public function findForSitemap()
    {
        return $this->createQueryBuilder('n')
            ->where('n.status = :status')
            ->andWhere('n.dateDeleted IS NULL')
            ->andWhere('n.dateOfPublication <= :dateOfPublication')
            ->setParameters([
                'status' => 'A',
                'dateOfPublication' => new \DateTime(),
            ])
            ->orderBy('n.dateOfPublication', 'DESC')
            ->getQuery()
            ->getResult()
        ;
    }

    // -------------------------------------------------------------------------

    /**
     * Pobiera ulubione aktualności użytkownika.
     *
     * @param User $user
     * @return array
     */
    public function findFavoriteNewsByUser(User $user) : array
    {
        return $this->createQueryBuilder('n')
            ->leftJoin('n.users', 'u')
            ->where('u.id = :user_id')
            ->andWhere('n.type = :type')
            ->setParameters([
                'user_id' => $user->getId(),
                'type' => News::TYPE_NEWS,
            ])
            ->getQuery()
            ->getResult()
        ;
    }

    /**
     * Pobiera ulubione wpisy użytkownika.
     *
     * @param User $user
     * @return array
     */
    public function findFavoritePostsByUser(User $user) : array
    {
        return $this->createQueryBuilder('n')
            ->leftJoin('n.users', 'u')
            ->where('u.id = :user_id')
            ->andWhere('n.type = :type')
            ->setParameters([
                'user_id' => $user->getId(),
                'type' => News::TYPE_BLOG,
            ])
            ->getQuery()
            ->getResult()
        ;
    }

}
