<?php

namespace Wi\Admin\NewsLetterBundle\Controller;

use Knp\Component\Pager\Paginator;
use Symfony\Bundle\FrameworkBundle\Console\Application;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\Console\Input\ArrayInput;
use Symfony\Component\Console\Output\NullOutput;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\KernelInterface;
use Wi\Admin\NewsLetterBundle\Entity\Campaign;
use Wi\Admin\NewsLetterBundle\Entity\Config;

/**
 * Mainsite Newsletter controller.
 *
 * @author Maksymilian Baryczka <mmaxioo33@gmail.com>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class MainsiteController extends Controller
{
    /**
     * Główny widok zarządzania newsletterem.
     *
     * @param Request $request
     * @param Paginator $paginator
     * @return Response
     */
    public function index2Action(Request $request, Paginator $paginator)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $configRepo = $em->getRepository('WiAdminNewsLetterBundle:Config');

        // Pobranie wszystkich baz relacyjnych.
        $messageContent = $em->getRepository('WiAdminNewsLetterBundle:MessageContent')->findAllActive();
        $recipients = $em->getRepository('WiAdminNewsLetterBundle:Recipients')->findAllActive();
        $senders = $em->getRepository('WiAdminNewsLetterBundle:Senders')->findAllActive();
        $campaigns = $em->getRepository('WiAdminNewsLetterBundle:Campaign')->findAllActive();

        $enableNewsletter = $configRepo->getCurrentConfigByParamName('enable_newsletter');
        $quantityToSent = $configRepo->getCurrentConfigByParamName('quantity_to_sent');
        $currentCampaign = $configRepo->getCurrentConfigByParamName('current_campaign');
        $dateStart = $configRepo->getCurrentConfigByParamName('date_start');
        $dateEnd = $configRepo->getCurrentConfigByParamName('date_end');
        $randomizeSenders = $configRepo->getCurrentConfigByParamName('randomize_senders');
        $randomizeRecipients = $configRepo->getCurrentConfigByParamName('randomize_recipients');
        $randomizeMessages = $configRepo->getCurrentConfigByParamName('randomize_messages');
        $bcc = $configRepo->getCurrentConfigByParamName('bcc');
        $daysOfWeek = $configRepo->getCurrentConfigByParamName('days_of_week');
        $timeEnd = $configRepo->getCurrentConfigByParamName('time_end');
        $timeStart = $configRepo->getCurrentConfigByParamName('time_start');
        $activeCampaign = $em->getRepository('WiAdminNewsLetterBundle:Campaign')->find($currentCampaign->getParamValue());

        // Wyłączenie filtra softdeleteable.
        $em->getFilters()->disable('softdeleteable');
        $logs = $paginator->paginate(
            $em->getRepository('WiAdminNewsLetterBundle:Logs')
                ->getLogsByCampaign($activeCampaign->getId()),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 100
        );
        // Włączenie filtra softdeleteable.
        $em->getFilters()->enable('softdeleteable');

        // Liczba wysłanych wiadomosci w aktualnej kampanii.
        $sendedMessages = $activeCampaign->getLogs()->filter(function ($entry) {
            return $entry->getStatus() == 'A';
        });
        // Liczba odbiorców, którzy wyrazili zgodę na wysyłanie newsletter-a.
        $recipientsWithConsent = $recipients->filter(function ($entry) {
            return $entry->getConsent() == 1;
        })->count();
        // Liczba wiadomości, które pozostały do wysłania.
        $awaitingMessages = ($count = $recipientsWithConsent - $sendedMessages->count()) > 0 ? $count : 0;

        if ($campaigns) {
            foreach ($campaigns as $singleCampaign) {
                if ($singleCampaign->getId() == $currentCampaign->getParamValue()) {
                    // Date start for range
                    $infoTab[1] = $singleCampaign->getDateCreated();
                    // Date end for range
                    $infoTab[2] = $singleCampaign->getDateExpired();
                    // Send messages counter
                    $infoTab[3] = count($singleCampaign->getMessageContent());
                    // Current campaign name
                    $infoTab[6] = $singleCampaign->getName();
                }
            }
        } else {
            // Date start for range
            $infoTab[1] = '';
            // Date end for range
            $infoTab[2] = '';
            // Send messages counter
            $infoTab[3] = '';
            // Current campaign name
            $infoTab[6] = '';
        }

        // Senders counter
        $infoTab[5] = count($senders);

        if (is_null($enableNewsletter) || ! $enableNewsletter->getParamValue()) {
            $infoTab[7] = ('Newsletter wylaczony.');
        } elseif (! $dateStart->getParamValue() || ! $dateEnd->getParamValue() || ! $daysOfWeek->getParamValue() || ! $timeEnd->getParamValue() || ! $timeStart->getParamValue()) {
            $infoTab[7] = 'Data nie pozwala na uruchomienie newslettera.';
        } elseif (! $currentCampaign->getParamValue() || empty($campaigns)) {
            $infoTab[7] = 'Kampania nie zostala wybrana.';
        } elseif ($infoTab[2]) {
            if ($infoTab[2]->format('Y-m-d') < (new \DateTime())->format('Y-m-d')) {
                $infoTab[7] = 'Kampania stracila waznosc.';
            }
        } elseif ($quantityToSent->getParamValue() <= 0) {
            $infoTab[7] = 'Liczba wiadomosci do wyslania jest ustawiona na 0.';
        } else {
            $infoTab[7] = 'Aktywny';
        }

        return $this->render('WiAdminNewsLetterBundle:Mainsite:index.html.php', [
            'activeCampaign' => $activeCampaign,
            'awaitingMessages' => $awaitingMessages,
            'bcc' => $bcc,
            'campaigns' => $campaigns,
            'currentCampaign' => $currentCampaign,
            'dateEnd' => $dateEnd,
            'dateStart' => $dateStart,
            'daysOfWeek' => $daysOfWeek,
            'enableNewsletter' => $enableNewsletter,
            'infoTab' => $infoTab,
            'langs' => [
                'messages' => $this->getLangMessages($activeCampaign),
                'senders' => $this->getLangSenders($senders),
            ],
            'logs' => $logs,
            'messageContent' => $messageContent,
            'quantityToSent' => $quantityToSent,
            'randomizeMessages' => $randomizeMessages,
            'randomizeRecipients' => $randomizeRecipients,
            'randomizeSenders' => $randomizeSenders,
            'recipients' => $recipients,
            'sendedMessages' => $sendedMessages,
            'senders' => $senders,
            'timeEnd' => $timeEnd,
            'timeStart' => $timeStart,
        ]);
    }

    /**
     * Ajax updating config params with soft delete.
     *
     * @return Response
     */
    public function ajaxsAction()
    {
        if (isset($_POST['paramName'])) {
            $paramName = $_POST['paramName'];
        }

        if (isset($_POST['paramValue'])) {
            $paramValue = rtrim($_POST['paramValue'],',');
        } else {
            $paramValue=null;
        }

        if (isset($_POST['aidValue'])) {
            $aidValue = $_POST['aidValue'];
        } else {
            $aidValue=null;
        }

        if (isset($_POST['id'])){
            $id = $_POST['id'];
        }

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        $oldParam = $em->getRepository('WiAdminNewsLetterBundle:Config')->findOneById($id);

        // Usunięcie.
        $em->remove($oldParam);

        $newParam = new Config();
        $newParam
            ->setParamName($paramName)
            ->setParamValue($paramValue)
            ->setAidValue($aidValue)
        ;
        $em->persist($newParam);
        $em->flush();

        return new Response(1);
    }

    /**
     * Manual start mailing.
     *
     * @param KernelInterface $kernel
     * @return Response
     */
    public function manualStartAction(KernelInterface $kernel, Campaign $campaign = null)
    {
        $inputArr = [
            'command' => 'app:cron:newsletters',
        ];

        if (! is_null($campaign)) {
            $inputArr['--campaign'] = $campaign->getId();
        }

        $application = new Application($kernel);
        $application->setAutoExit(false);
        $application->run(
            new ArrayInput($inputArr),
            new NullOutput()
        );

        if (! is_null($campaign)) {
            return $this->redirectToRoute('wi_admin_newsletter_campaign_index', [
                'id' => $campaign->getId(),
            ]);
        }

        return $this->redirectToRoute('wi_admin_newsletter_mainsite_index');
    }

    public function statisticsAction(Request $request, Campaign $campaign)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $type = $request->query->get('type');
        $out = [
            'data' => [],
        ];

        switch ($type) {
            case 'daily':
                $date = ($date = $request->query->get('date')) ? (new \DateTime($date)) : (new \DateTime());

                if ($date > $campaign->getDateExpired()) {
                    $date = $campaign->getDateExpired();
                }

                $daily = $em->getRepository('WiAdminNewsLetterBundle:ViewNewsletterStatsPerHour')->findBy([
                    'campaignId' => $campaign->getId(),
                    'date' => $date->format('Y-m-d'),
                ]);
                $next = clone $date;
                $prev = clone $date;

                $out['next_day'] = $next->modify('+1 day')->format('Y-m-d');
                $out['prev_day'] = $prev->modify('-1 day')->format('Y-m-d');

                for ($i = 0; $i < 24; $i++) {
                    $out['data']["$i"] = ["$i", 0, ''];
                }

                foreach ($daily as $hour) {
                    $out['data'][''.(int) $hour->getHour().''] = [
                        ''.(int) $hour->getHour().'',
                        (int) $hour->getMessages(),
                        '<div style="padding:5px;white-space:nowrap;">Hour: '.$hour->getHour().':00<br>Sent: '.(int) $hour->getMessages().' messages.</div>'
                    ];
                }

                $out['curr_day'] = $date->format('Y-m-d');
                break;
            case 'monthly':
                $date = ($date = $request->query->get('date')) ? (new \DateTime($date)) : (new \DateTime());

                if ($date > $campaign->getDateExpired()) {
                    $date = $campaign->getDateExpired();
                }

                $monthly = $em->getRepository('WiAdminNewsLetterBundle:ViewNewsletterStatsPerDay')
                    ->findByMonth($campaign->getId(), $date->format('Y-m'))
                ;
                $next = clone $date;
                $prev = clone $date;

                $out['next_month'] = $next->modify('+1 month')->format('Y-m');
                $out['prev_month'] = $prev->modify('-1 month')->format('Y-m');

                for ($i = 0; $i < $date->format('t'); $i++) {
                    $out['data']["$i"] = [''.($i+1).'', 0, ''];
                }

                foreach ($monthly as $day) {
                    $dayNumber = (new \DateTime($day->getDate()))->format('d');
                    $out['data'][''.(intval($dayNumber) - 1).''] = [
                        ''.(int) $dayNumber.'',
                        (int) $day->getMessages(),
                        '<div style="padding:5px;white-space:nowrap;">Day: '.intval($dayNumber).'<br>Sent: '.(int) $day->getMessages().' messages.</div>'
                    ];
                }

                $out['curr_month'] = $date->format('Y-m');
                break;
            case 'campaign':
                $all = $em->getRepository('WiAdminNewsLetterBundle:ViewNewsletterStatsPerDay')->findByCampaignId($campaign->getId());
                $dateStart = clone $campaign->getDateCreated();
                $dateEnd = $campaign->getDateExpired();

                while ($dateStart <= $dateEnd) {
                    $out['data'][$dateStart->format('Y-m-d')] = [
                        'date' => $dateStart->format('Y-m-d'),
                        'messages' => 0,
                        'tooltip' => '',
                    ];
                    $dateStart->modify('+1 day');
                }

                foreach ($all as $one) {
                    $out['data'][$one->getDate()] = [
                        'date' => $one->getDate(),
                        'messages' => (int) $one->getMessages(),
                        'tooltip' => '<div style="padding:5px;white-space:nowrap;">Day: '.$one->getDate().'<br>Sent: '.(int) $one->getMessages().' messages.</div>',
                    ];
                }

                $out['data'] = array_values($out['data']);

                break;
            default:
                break;
        }

        return new JsonResponse($out);
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Przetowrzenie listy języków wiadomości.
     *
     * @param Campaign|null $campaign
     * @return string
     */
    private function getLangMessages($campaign)
    {
        $tmp = [];

        if (! is_null($campaign)) {
            foreach ($campaign->getMessageContent() as $messageContent) {
                $tmp[$messageContent->getLang()] = null;
            }
        }

        ksort($tmp);

        return strtoupper(implode(',', array_keys($tmp)));
    }

    /**
     * Przetworzenie listy języków nadawców.
     *
     * @param array $senders
     * @return string
     */
    private function getLangSenders($senders)
    {
        $tmp = [];

        foreach ($senders as $sender) {
            $tmp[$sender->getLang()] = null;
        }

        ksort($tmp);

        return strtoupper(implode(',', array_keys($tmp)));
    }
}
