<?php

namespace Wi\Admin\NewsLetterBundle\Controller;

use Knp\Component\Pager\Paginator;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\Form\Extension\Core\Type\FileType;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Wi\Admin\NewsLetterBundle\Entity\RecipientGroup;
use Wi\Admin\NewsLetterBundle\Entity\Recipients;
use Wi\Admin\NewsLetterBundle\Form\DeleteType;
use Wi\Admin\NewsLetterBundle\Form\RecipientGroupType;
use Wi\Admin\NewsLetterBundle\Service\HashRecipientGroup;
use Wi\Admin\NewsLetterBundle\Service\SignUpCodeGenerator;

/**
 * Recipient group controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class RecipientGroupController extends Controller
{
    public function indexAction()
    {
        $em = $this->getDoctrine()->getManager();
        $recipientGroups = $em->getRepository('WiAdminNewsLetterBundle:RecipientGroup')->findAll();

        return $this->render('WiAdminNewsLetterBundle:RecipientGroup:index.html.php', [
            'recipientGroups' => $recipientGroups,
        ]);
    }

    public function newAction(Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $recipientGroup = new RecipientGroup();
        $campaign = $em->getRepository('WiAdminNewsLetterBundle:Campaign')->find($request->query->getInt('cid'));

        $form = $this->createForm(RecipientGroupType::class, $recipientGroup);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            if (! is_null($campaign)) {
                $campaign->addRecipientGroup($recipientGroup);
                $em->persist($campaign);
            }

            $em->persist($recipientGroup);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_recipient_group_show', [
                'id' => $recipientGroup->getId(),
            ]);
        }

        return $this->render('WiAdminNewsLetterBundle:RecipientGroup:new.html.php', [
            'form' => $form->createView(),
            'recipientGroup' => $recipientGroup,
        ]);
    }

    public function showAction(Request $request, Paginator $paginator, RecipientGroup $recipientGroup)
    {
        $em = $this->getDoctrine()->getManager();
        $recipients = $paginator->paginate(
            $em->getRepository('WiAdminNewsLetterBundle:Recipients')->getFindAllByGroupQuery($recipientGroup),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 100,
            [
                'defaultSortDirection' => 'ASC',
                'defaultSortFieldName' => 'r.email',
            ]
        );

        $form = $this->createFormBuilder()
            ->add('file', FileType::class, [
                'attr' => [
                    'accept' => '.csv',
                    'class' => 'InputFile',
                ],
                'label' => false,
                'mapped' => false,
                'required' => true,
            ])
            ->getForm()
        ;
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $file = $form->get('file')->getData();

            if (($handle = fopen($file, 'r')) !== false) {
                while (($row = fgetcsv($handle)) !== false) {
                    $email = ! empty($row[0]) ? $row[0] : '';
                    $name = ! empty($row[1]) ? $row[1] : null;
                    $consent = ! empty($row[2]) ? boolval(intval($row[2])) : false;
                    $notUnique = (bool) $em->getRepository('WiAdminNewsLetterBundle:Recipients')->findOneBy([
                        'email' => $email,
                        'recipientGroup' => $recipientGroup,
                    ]);

                    if (! $notUnique) {
                        $recipient = new Recipients();
                        $recipient
                            ->setEmail($email)
                            ->setRecipientGroup($recipientGroup)
                            ->setName($name)
                            ->setConsent($consent)
                        ;
                        $em->persist($recipient);
                    }
                }

                $em->flush();
            }

            return $this->redirectToRoute('wi_admin_newsletter_recipient_group_show', [
                'id' => $recipientGroup->getId(),
            ]);
        }

        return $this->render('WiAdminNewsLetterBundle:RecipientGroup:show.html.php', [
            'form' => $form->createView(),
            'recipientGroup' => $recipientGroup,
            'recipients' => $recipients,
        ]);
    }

    public function deleteAction(Request $request, RecipientGroup $recipientGroup)
    {
        if ($recipientGroup->getId() == 1) {
            return $this->redirectToRoute('wi_admin_newsletter_recipient_group_show', [
                'id' => $recipientGroup->getId()
            ]);
        }

        // Generowanie formularza.
        $deleteForm = $this->createForm(DeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usunięcie.
            $em->remove($recipientGroup);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_recipient_group_index');
        }

        return $this->render('WiAdminNewsLetterBundle:RecipientGroup:delete.html.php', [
            'deleteForm' => $deleteForm->createView(),
            'recipientGroup' => $recipientGroup,
        ]);
    }

    public function multipleAction(Request $request)
    {
        $ids = $request->request->get('ids');
        $option = $request->request->get('option');
        $em = $this->getDoctrine()->getManager();
        $recipients = $em->getRepository('WiAdminNewsLetterBundle:Recipients')->findById($ids);

        switch ($option) {
            case 'consent-0': // Zabronienie wysyłania wiadomości.
                foreach ($recipients as $recipient) {
                    $recipient->setConsent(0);
                }

                $em->flush();
                break;
            case 'consent-1': // Zezwolenie na wysyłanie wiadomości.
                foreach ($recipients as $recipient) {
                    $recipient->setConsent(1);
                }

                $em->flush();
                break;
            case 'delete': // Usuwanie.
                foreach ($recipients as $recipient) {
                    $em->remove($recipient);
                }

                $em->flush();
                break;
        }

        return new JsonResponse([
            'status' => true,
        ]);
    }

    public function formCodeAction(
        Request $request,
        HashRecipientGroup $hasher,
        SignUpCodeGenerator $generator,
        RecipientGroup $recipientGroup
    ) {
        $id = $hasher->encode($recipientGroup->getId());
        $code = $generator->generate($id, (bool) $request->request->getInt('add_name'));

        return new JsonResponse([
            'code' => $code,
        ]);
    }
}
