<?php

namespace Wi\Admin\PageBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\PageBundle\Entity\PageGroup;
use Wi\Admin\PageBundle\Form\PageGroupDeleteType;
use Wi\Admin\PageBundle\Form\PageGroupType;

/**
 * PageGroup controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class PageGroupController extends Controller
{
    /**
     * Lists all pageGroup entities.
     *
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction()
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich głównych, aktywnych grup stron.
        $pageGroups = $em->getRepository('WiAdminPageBundle:PageGroup')
            ->findAllActiveWithoutParent()
        ;

        return $this->render('WiAdminPageBundle:PageGroup:index.html.php', [
            'pageGroups' => $pageGroups,
        ]);
    }

    /**
     * Creates a new pageGroup entity.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        // Utworzenie nowego obiektu grupy.
        $pageGroup = new PageGroup();

        // Ustawienie kolejności.
        $pageGroup->setOrderNr(0);

        // Generowanie formularza.
        $form = $this->createForm(PageGroupType::class, $pageGroup);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($pageGroup);
            $em->flush();

            return $this->redirectToRoute('wi_admin_pages_index');
        }

        return $this->render('WiAdminPageBundle:PageGroup:new.html.php', [
            'form' => $form->createView(),
            'pageGroup' => $pageGroup,
        ]);
    }

    /**
     * Finds and displays a pageGroup entity.
     *
     * @param   Request $request
     * @param   PageGroup $pageGroup
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(PageGroup $pageGroup)
    {
        return $this->render('WiAdminPageBundle:PageGroup:show.html.php', [
            'pageGroup' => $pageGroup,
        ]);
    }

    /**
     * Displays a form to edit an existing pageGroup entity.
     *
     * @param   Request $request
     * @param   PageGroup $pageGroup
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, PageGroup $pageGroup)
    {
        // Generowanie formularza.
        $editForm = $this->createForm(PageGroupType::class, $pageGroup);
        $editForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($editForm->isSubmitted() && $editForm->isValid()) {
            // Zapis.
            $this->getDoctrine()->getManager()->flush();

            return $this->redirectToRoute('wi_admin_page_groups_show', [
                'id' => $pageGroup->getId(),
            ]);
        }

        return $this->render('WiAdminPageBundle:PageGroup:edit.html.php', [
            'edit_form' => $editForm->createView(),
            'pageGroup' => $pageGroup,
        ]);
    }

    /**
     * Deletes a pageGroup entity.
     *
     * @param   Request $request
     * @param   PageGroup $pageGroup
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, PageGroup $pageGroup)
    {
        // Utworzenie formularza usuwania grup stron.
        $formDelete = $this->createForm(PageGroupDeleteType::class, $pageGroup);
        $formDelete->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($formDelete->isSubmitted() && $formDelete->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            if ($pageGroup->getPages()->count() || $pageGroup->getChildren()->count()) {
                // Usuwanie wszystkich stron i grup.
                if ($formDelete->get('deleteAll')->getData()) {
                    // Rekursywne usuwanie grup.
                    $recursiveRemove = function ($pageGroup) use (&$recursiveRemove, $em) {
                        if ($pageGroup->getChildren()->count() > 0) {
                            foreach ($pageGroup->getChildren() as $child) {
                                $recursiveRemove($child);
                            }
                        }

                        // Usuwanie stron.
                        foreach ($pageGroup->getPages() as $page) {
                            $em->remove($page);
                            $em->flush();
                        }

                        $em->remove($pageGroup);
                    };

                    $recursiveRemove($pageGroup);
                } else {
                    // Usunięcie grup i przenoszenie stron i podgrup.
                    $newPageGroup = $formDelete->get('newPageGroup')->getData();

                    foreach ($pageGroup->getPages() as $page) {
                        // Dodanie nowej kategorii.
                        $page->setGroup($newPageGroup);

                        // Przygotowanie do zapisu.
                        $em->persist($page);
                    }

                    // Przypisanie nowego rodzica podgrupom.
                    foreach ($pageGroup->getChildren() as $child) {
                        // Ustawienie nowego rodzica.
                        $child->setParent($newPageGroup);

                        // Przygotowanie do zapisu.
                        $em->persist($child);
                    }

                    // Zapis edytowanych stron i podgrup.
                    $em->flush();
                }
            }

            // Usuwanie grupy.
            $em->remove($pageGroup);
            $em->flush();

            // Ustawienie daty usunięcia w grup ze statusem 'D'. Przy usuwaniu
            // jest problem ustawić dlatego ten kod się pojawił.
            $deletedPageGroups = $em->getRepository('WiAdminPageBundle:PageGroup')
                ->findBy([
                    'dateDeleted' => null,
                    'status' => 'D',
                ])
            ;

            if (count($deletedPageGroups) > 0) {
                foreach ($deletedPageGroups as $deletedPageGroup) {
                    $deletedPageGroup->setDateDeleted(new \DateTime());
                    $em->persist($deletedPageGroup);
                }

                $em->flush();
            }

            return $this->redirectToRoute('wi_admin_page_groups_index');
        }

        return $this->render('WiAdminPageBundle:PageGroup:delete.html.php', [
            'formDelete' => $formDelete->createView(),
            'pageGroup' => $pageGroup,
        ]);
    }

    /**
     * Update a order pageGroup entity.
     *
     * @param PageGroup $pageGroup
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function updateOrderAction(PageGroup $pageGroup, $sort_dir)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $pageGroupRepo = $em->getRepository('WiAdminPageBundle:PageGroup');

        // Pobranie wszystkich aktywnych grup.
        $pageGroups = $pageGroupRepo->findAllActiveWithoutParent();

        // Poprawienie kolejności. Może wystąpić sytuacja, że kilka grup
        // będzie miało przypisany ten sam numer w kolejności i wtedy
        // przenoszenie o '1' nie spełnia swojej roli.
        $fixOrder = function ($pageGroups, $order = 1) use (&$fixOrder, $em) {
            foreach ($pageGroups as $pageGroup) {
                $pageGroup->setOrderNr($order++);

                $em->persist($pageGroup);

                if ($pageGroup->getChildren()->count()) {
                    $fixOrder($pageGroup->getChildren(), 1);
                }
            }

            $em->flush();
        };

        $fixOrder($pageGroups);

        switch ($sort_dir) {
            case 'up': // Przesunięcie w górę.
                $nextTo = $pageGroupRepo->findOneBy([
                    'orderNr' => $pageGroup->getOrderNr() - 1,
                    'parent' => $pageGroup->getParent(),
                ]);

                if (! is_null($nextTo)) {
                    $nextTo->setOrderNr($pageGroup->getOrderNr());
                    $pageGroup->setOrderNr($pageGroup->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            case 'down': // Przesunięcie w dół.
                $nextTo = $pageGroupRepo->findOneBy([
                    'orderNr' => $pageGroup->getOrderNr() + 1,
                    'parent' => $pageGroup->getParent(),
                ]);

                if (! is_null($nextTo)) {
                    $pageGroup->setOrderNr($nextTo->getOrderNr());
                    $nextTo->setOrderNr($nextTo->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            default:
                break;
        }

        $em->persist($pageGroup);
        $em->flush();

        return $this->redirectToRoute('wi_admin_page_groups_index');
    }
}
