<?php

namespace Wi\Admin\UserBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\UserBundle\Entity\BillingData;
use Wi\Admin\UserBundle\Entity\Profile;
use Wi\Admin\UserBundle\Entity\User;
use Wi\Admin\UserBundle\Form\ProfileType;
use Wi\Admin\UserBundle\Form\UserNewsletterType;
use Wi\Admin\UserBundle\Form\UserStatusType;
use Wi\Admin\UserBundle\Form\UserType;
use Wi\Bidbee\WalletBundle\Entity\VirtualTransaction;
use Wi\Bidbee\WalletBundle\Entity\Wallet;

/**
 * User controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class UserController extends Controller
{
    /**
     * List of User entities.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich użytkowników.
        $paginator = $this->get('knp_paginator');
        $users = $paginator->paginate(
            $em->getRepository('WiAdminUserBundle:User')
                ->getFindAllQuery($request->query->get('q')),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 10,
            [
                'wrap-queries' => true,
            ]
        );

        return $this->render('WiAdminUserBundle:User:index.html.php', [
            'users' => $users,
        ]);
    }

    /**
     * Show a user entity.
     *
     * @param   Request $request
     * @param   User    $user
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Request $request, User $user)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich użytkowników.
        $paginator = $this->get('knp_paginator');
        $logs = $paginator->paginate(
            $em->getRepository('WiAdminUserBundle:Log')
                ->getLogsByUserQuery($user, $request->query->get('q')),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 10,
            [
                'wrap-queries' => true,
            ]
        );
        $favoriteNews = $em->getRepository('WiAdminNewsBundle:News')
            ->findFavoriteNewsByUser($user)
        ;
        $favoritePosts = $em->getRepository('WiAdminNewsBundle:News')
            ->findFavoritePostsByUser($user)
        ;

        return $this->render('WiAdminUserBundle:User:show.html.php', [
            'favoriteNews' => $favoriteNews,
            'favoritePosts' => $favoritePosts,
            'logs' => $logs,
            'user' => $user,
        ]);
    }

    /**
     * Edit a user entity.
     *
     * @param   Request $request
     * @param   User    $user
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, User $user)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Generowanie formularza edycji zmiany hasła użytkownika.
        $userForm = $this->createForm(UserType::class, $user);
        $userForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($userForm->isSubmitted() && $userForm->isValid()) {
            // Ustawienie hasła.
            $user->setPassword(
                $this->get('security.password_encoder')
                    ->encodePassword(
                        $user,
                        $user->getPassword()
                    )
            );

            // Zapis.
            $em->persist($user);
            $em->flush();

            return $this->redirectToRoute('wi_admin_users_show', [
                'id' => $user->getId(),
            ]);
        }

        // Generowanie formularza zmiany statusu użytkownika.
        $userStatusForm = $this->createForm(UserStatusType::class, $user);
        $userStatusForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($userStatusForm->isSubmitted() && $userStatusForm->isValid()) {
            // Zapis.
            $em->persist($user);
            $em->flush();

            return $this->redirectToRoute('wi_admin_users_show', [
                'id' => $user->getId(),
            ]);
        }

        // Generowanie formularza zmiany zgody użytkownika na mail'e.
        $userNewsletterForm = $this->createForm(UserNewsletterType::class, $user);
        $userNewsletterForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($userNewsletterForm->isSubmitted() && $userNewsletterForm->isValid()) {
            // Zapis.
            $em->persist($user);
            $em->flush();

            return $this->redirectToRoute('wi_admin_users_show', [
                'id' => $user->getId(),
            ]);
        }

        // Sprawdzenie, czy istnieje obiekt danych profilowych użytkownika.
        if (! $profile = $user->getProfile()->first()) {
            // Utworzenie nowego obiektu danych profilowych.
            $profile = new Profile();

            // Przypisanie użytkownika.
            $profile->setUser($user);
        }

        // Generowanie formularza edycji danych profilowych.
        $profileForm = $this->createForm(ProfileType::class, $profile);
        $profileForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($profileForm->isSubmitted() && $profileForm->isValid()) {
            if ($lastProfile = $user->getProfile()->first()) {
                // Usunięcie starych danych profilowych.
                $em->remove($lastProfile);

                // Utworzenie nowych danych profilowych.
                $newProfile = new Profile();
                $newProfile
                    ->setFirstName($profile->getFirstName())
                    ->setLastName($profile->getLastName())
                    ->setSex($profile->getSex())
                    ->setDateOfBirth($profile->getDateOfBirth())
                    ->setDescription($profile->getDescription())
                    ->setPesel($profile->getPesel())
                    ->setUser($user)
                ;
                $profile = $newProfile;
            }

            // Dodanie prolifu do użytkownika.
            $user->addProfile($profile);

            // Zapis.
            $em->persist($profile);
            $em->flush();

            return $this->redirectToRoute('wi_admin_users_show', [
                'id' => $user->getId(),
            ]);
        }

        // Sprawdzenie, czy istnieje obiekt danych billingowych użytkownika.
        if (! $billingData = $user->getBillingData()->first()) {
            // Utworzenie nowego obiektu danych billingowych.
            $billingData = new BillingData();

            // Przypisanie użytkownika.
            $billingData->setUser($user);

        }

        // Generowanie formularza edycji danych billingowych.
        $billingDataForm = $this->createForm('Wi\Admin\UserBundle\Form\BillingDataType', $billingData);
        $billingDataForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($billingDataForm->isSubmitted() && $billingDataForm->isValid()) {
            // Remove white space from strings.
            $billingData->setZip(preg_replace('/\s/', '', $billingData->getZip()));
            $billingData->setNIP(preg_replace('/\s/', '', $billingData->getNIP()));

            if ($lastBillingData = $user->getBillingData()->first()) {
                // Usunięcie starych danych billingowych.
                $em->remove($lastBillingData);

                // Utworzenie nowych danych billingowych.
                $newBillingData = new BillingData();
                $newBillingData
                    ->setName($billingData->getName())
                    ->setAddress($billingData->getAddress())
                    ->setZip($billingData->getZip())
                    ->setCity($billingData->getCity())
                    ->setCountry($billingData->getCountry())
                    ->setNIP($billingData->getNIP())
                    ->setUser($user)
                ;
                $billingData = $newBillingData;

            }

            // Dodanie prolifu do użytkownika.
            $user->addBillingData($billingData);

            // Zapis.
            $em->persist($billingData);
            $em->flush();

            return $this->redirectToRoute('wi_admin_users_show', [
                'id' => $user->getId(),
            ]);
        }

        return $this->render('WiAdminUserBundle:User:edit.html.php', [
            'billingDataForm' => $billingDataForm->createView(),
            'profileForm' => $profileForm->createView(),
            'userForm' => $userForm->createView(),
            'userNewsletterForm' => $userNewsletterForm->createView(),
            'userStatusForm' => $userStatusForm->createView(),
            'user' => $user,
        ]);
    }
}
