<?php

namespace Wi\Admin\UserBundle\User;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use Wi\Admin\UserBundle\Entity\User;
use Wi\Front\UserBundle\Utils\Logger;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Front\UserBundle\Service\Mailer;

class UserManager
{
    private $config;
    private $em;
    private $logger;
    private $mailer;
    private $passwordEncoder;
    private $requestStack;
    private $session;

    private $user;

    public function __construct(
        Config $config,
        EntityManagerInterface $em,
        Logger $logger,
        Mailer $mailer,
        RequestStack $requestStack,
        SessionInterface $session,
        UserPasswordEncoderInterface $passwordEncoder
    ) {
        $this->config = $config;
        $this->em = $em;
        $this->logger = $logger;
        $this->mailer = $mailer;
        $this->passwordEncoder = $passwordEncoder;
        $this->requestStack = $requestStack;
        $this->session = $session;
    }

    public function register(User $user) : bool
    {
        $this
            ->setUser($user)
            ->setHashCode()
            ->encodePassword()
            ->setRegisterStatus()
        ;

        $this->save();
        $this->logCreate();

        if ((! $this->confirmEmail() && $this->autoAccept()) || (! $this->confirmEmail() && ! $this->autoAccept())) { // aktywny
            $this->notifyAdministrators();
        }

        if ($this->confirmEmail()) {
            if ($this->sendActivationEmail()) {
                return true;
            }

            return false;
        }

        return true;
    }

    public function activate(User $user) : bool
    {
        $this
            ->setUser($user)
            ->clearHashCode()
            ->setActivateStatus()
        ;

        $this->logActivation();

        if ($this->confirmEmail()) {
            $this->notifyAdministrators();
        }

        return true;
    }

    // -------------------------------------------------------------------------

    private function save()
    {
        $this->em->persist($this->user);
        $this->em->flush($this->user);
    }

    private function setUser(User $user)
    {
        $this->user = $user;

        return $this;
    }

    private function setHashCode()
    {
        $this->user->setHashCode(
            hash('sha256', md5(
                time()
                . $this->session->getId()
                . $this->user->getEmail()
                . time()
                . rand()
            ))
        );

        return $this;
    }

    private function clearHashCode()
    {
        $this->user->setHashCode(null);

        return $this;
    }

    private function encodePassword()
    {
        $this->user->setPassword(
            $this->passwordEncoder
                ->encodePassword($this->user, $this->user->getPassword())
        );

        return $this;
    }

    private function setActivateStatus()
    {
        if ($this->confirmEmail() && ! $this->autoAccept()) { // zablokowany do momentu potwierdzenia adresu email - po potwierdzeniu nieaktywny
            $this->setStatusInactive();
        } else {
            $this->setStatusActive();
        }

        return $this;
    }

    private function setRegisterStatus()
    {
        if ($this->confirmEmail() && $this->autoAccept()) { // zablokowany do momentu potwierdzenia adresu email - po potwierdzeniu aktywny
            $this->setStatusBlocked();
        } else if (! $this->confirmEmail() && $this->autoAccept()) { // aktywny
            $this->setStatusActive();
        } else if ($this->confirmEmail() && ! $this->autoAccept()) { // zablokowany do momentu potwierdzenia adresu email - po potwierdzeniu nieaktywny
            $this->setStatusBlocked();
        } else if (! $this->confirmEmail() && ! $this->autoAccept()) { // nieaktywny
            $this->setStatusInactive();
        } else {
            throw new \Exception('Cannot set the correct status to the user.');
        }

        return $this;
    }

    private function setStatusActive()
    {
        $this->user->setStatus(User::STATUS_ACTIVE);

        return $this;
    }

    private function setStatusBlocked()
    {
        $this->user->setStatus(User::STATUS_BLOCKED);

        return $this;
    }

    private function setStatusInactive()
    {
        $this->user->setStatus(User::STATUS_INACTIVE);

        return $this;
    }

    // -------------------------------------------------------------------------

    private function notifyAdministrators()
    {
        if ($this->notifyAdmin()) {
            if ($emails = $this->notifyAdminList()) {
                return $this->mailer->sendNotifyActivationToAdmin($this->user, $emails);
            }
        }
    }

    private function sendActivationEmail() : int
    {
        return $this->mailer->sendActivationEmail($this->user);
    }

    private function logCreate()
    {
        $request = $this->requestStack->getCurrentRequest();

        $this->logger->create(1, [
            'user_id' => $this->user->getId(),
            'ip' => $request->server->get('REMOTE_ADDR'),
            'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
            'descriptions' => 'Account has been created.',
        ]);
    }

    private function logActivation()
    {
        $request = $this->requestStack->getCurrentRequest();

        $this->logger->create(2, [
            'user_id' => $this->user->getId(),
            'ip' => $request->server->get('REMOTE_ADDR'),
            'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
            'descriptions' => 'Email address has been verified.',
        ]);
    }

    // -------------------------------------------------------------------------

    private function autoAccept() : bool
    {
        return boolval(intval($this->config->get('user.autoAccept')));
    }

    private function confirmEmail() : bool
    {
        return boolval(intval($this->config->get('user.confirmEmail')));
    }

    private function notifyAdmin() : bool
    {
        return boolval(intval($this->config->get('user.notifyAdmin')));
    }

    private function notifyAdminList() : array
    {
        $administrators = $this->em
            ->getRepository('WiAdminAdminBundle:Administrator')
            ->findBy([
                'id' => explode(',', $this->config->get('user.notifyAdminList'))
            ])
        ;

        if (empty($administrators)) {
            return [];
        }

        $out = [];

        foreach ($administrators as $administrator) {
            $out[$administrator->getEmail()] = $administrator->getName();
        }

        return $out;
    }
}
