<?php

namespace Wi\Api\V1Bundle\Controller;

use FOS\RestBundle\Controller\Annotations\QueryParam;
use FOS\RestBundle\Controller\FOSRestController;
use FOS\RestBundle\Request\ParamFetcher;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\NewsBundle\Entity\News;
use Symfony\Component\HttpFoundation\Response;
use FOS\RestBundle\View\View;

/**
 * News controller for REST API.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class NewsController extends FOSRestController
{
    /**
     * Get single news.
     *
     * #### Example of a successful response ####
     * <details>
     *   <summary>Show details</summary>
     *
     *      {
     *        "id": 1,
     *        "title": "Lorem ipsum set dolor amet",
     *        "shortcut": "Lorem ipsum set dolor amet",
     *        "content": "<p>Lorem ipsum set dolor amet</p>",
     *        "dateOfPublication": "2017-08-22T00:00:00+02:00",
     *        "categories": [
     *          {
     *            "id": 1,
     *            "name": "Aktualności"
     *          }
     *        ],
     *        "images": {
     *          "original": "http://example.com/media/assets/News/example.jpg",
     *          "thumbnail": "http://example.com/media/assets/News/thumbnail/example.jpg",
     *          "thumbnailS": "http://example.com/media/assets/News/thumbnailS/example.jpg"
     *        }
     *      }
     * </details>
     *
     * #### Example of an invalid response ####
     * <details>
     *   <summary>Show details</summary>
     *
     *      {
     *        "error": {
     *          "code": 404,
     *          "message": "Not Found"
     *        }
     *      }
     * </details>
     *
     * @ApiDoc(
     *  description = "Get single news.",
     *  requirements = {
     *      {
     *          "name" = "id",
     *          "description" = "The news ID.",
     *          "dataType" = "integer",
     *          "requirement" = "\d+"
     *      }
     *  },
     *  resource = true,
     *  section = "News",
     *  statusCodes = {
     *      200 = "Returned when successfull.",
     *      404 = "Returned when not found.",
     *  },
     *  views = {"v1"}
     * )
     *
     * @param News $id
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function getNewAction(News $id)
    {
        $serializer = $this->get('serializer');
        $data = null;

        if ($id) {
            $data = $serializer->normalize($id, 'json', ['groups' => ['news']]);
        }

        return $this->handleView($this->view(
            $data,
            200
        ));
    }

    /**
     * Get list of news. The results are sorted default by date of publication.
     * For pagination to work, you must pass parameters: page and _per_page.
     *
     * #### Example of a successful response ####
     * <details>
     *   <summary>Show details</summary>
     *
     *      [
     *        {
     *          "id": 1,
     *          "title": "Lorem ipsum set dolor amet",
     *          "shortcut": "Lorem ipsum set dolor amet",
     *          "content": "<p>Lorem ipsum set dolor amet</p>",
     *          "dateOfPublication": "2017-08-22T00:00:00+02:00",
     *          "categories": [
     *            {
     *              "id": 1,
     *              "name": "Aktualności"
     *            }
     *          ],
     *          "images": {
     *            "original": "http://example.com/media/assets/News/example.jpg",
     *            "thumbnail": "http://example.com/media/assets/News/thumbnail/example.jpg",
     *            "thumbnailS": "http://example.com/media/assets/News/thumbnailS/example.jpg"
     *          }
     *        },
     *        {
     *          "id": 2,
     *          "title": "Consectetur sunt dignissimos aspernatur ab tempora",
     *          "shortcut": "Consectetur sunt dignissimos aspernatur ab tempora",
     *          "content": "<p>Consectetur sunt dignissimos aspernatur ab tempora</p>",
     *          "dateOfPublication": "2017-08-22T00:00:00+02:00",
     *          "categories": [
     *            {
     *              "id": 1,
     *              "name": "Aktualności"
     *            },
     *            {
     *              "id": 3,
     *              "name": "Polecane artykuły"
     *            }
     *          ],
     *          "images": {
     *            "original": "http://example.com/media/assets/News/example.jpg",
     *            "thumbnail": "http://example.com/media/assets/News/thumbnail/example.jpg",
     *            "thumbnailS": "http://example.com/media/assets/News/thumbnailS/example.jpg"
     *          }
     *        }
     *      ]
     * </details>
     *
     * @ApiDoc(
     *  description = "Get list of news.",
     *  resource = true,
     *  section = "News",
     *  statusCodes = {
     *      200 = "Returned when successfull.",
     *  },
     *  views = {"v1"}
     * )
     *
     * @QueryParam(name="_category_id", requirements="\d+", description="The news category ID.")
     * @QueryParam(name="_page", requirements="\d+", description="Page of the overview.")
     * @QueryParam(name="_per_page", requirements="\d+", description="Number of results per page.")
     * @QueryParam(name="_sort_dir", requirements="asc|desc", description="Sort direction.", default="desc")
     *
     * @param ParamFetcher $paramFetcher
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function getNewsAction(ParamFetcher $paramFetcher)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $serializer = $this->get('serializer');

        $news = $em->getRepository('WiAdminNewsBundle:News')
            ->findActivePublished(
                $paramFetcher->get('_category_id'),
                $paramFetcher->get('_page'),
                $paramFetcher->get('_per_page'),
                $paramFetcher->get('_sort_dir')
            )
        ;
        $totalCount = $em->getRepository('WiAdminNewsBundle:News')
            ->countActivePublished(
                $paramFetcher->get('_category_id')
            )
        ;
        $data = $serializer->normalize($news, 'json', ['groups' => ['news_list']]);
        $view = $this->view($data, 200)
            ->setHeader('X-Total-Count', $totalCount)
        ;

        return $this->handleView($view);
    }
    
    /**
    * Post one news item.
    * After successful API returns 201 (Created) code.
    * Added new news will be flagged as W (Waiting) upon administrator
    * accept it.
    * 
    * * <details>
    *   <summary>Show details</summary>
    *        {
    *          "title": "Some title",
    *          "content": "Lorem ipsum set dolor amet",
    *          "category_id": 1
    *        }
     * </details>
     * 
     * @ApiDoc(
     *  description = "Post new News item.",
     * requirements = {
     *      {
     *          "name" = "title",
     *          "description" = "The news title.",
     *          "dataType" = "string",
     *          "requirement" = "String, not empty"
     *      },
     *      {
     *          "name" = "content",
     *          "description" = "The news content body.",
     *          "dataType" = "string",
     *          "requirement" = "String, not empty"
     *      },
     *      {
     *          "name" = "category_id",
     *          "description" = "The Category ID.",
     *          "dataType" = "integer",
     *          "requirement" = "Integer, not empty"
     *      }
     *  },
     *  resource = true,
     *  section = "News",
     *  statusCodes = {
     *      201 = "Returned when created.",
     *  },
     *  views = {"v1"}
     * )
     *
     *
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function postNewAction(Request $request)
    {
        $data = new News();
        $title = $request->get('title');
        $content = $request->get('content');
        $category_id = $request->get('category_id');
        if($category_id == null) {
            $category_id = 1;
        }
        
        $em = $this->getDoctrine()->getManager();
        if (intval($category_id)) {
            $category = $em->getRepository('WiAdminNewsBundle:Category')
                ->findOneById(intval($category_id))
            ;

            if (! is_null($category)) {
                $data->addCategory($category);
            }
        }
        
        if (is_null($title) || is_null($content) || $category == null) {
            $view = $this->view('', 406);
            return $this->handleView($view);
        }
        $data
            ->setTitle($title)
            ->setContent($content)
            ->setDateOfPublication(new \DateTime())
            ->setStatus('W')
        ;
        
        $em->persist($data);
        $em->flush();
        
        $id = $data->getId();
        $serializer = $this->get('serializer');
        $data = $serializer->normalize($data, 'json', ['groups' => ['news']]);

        return $this->handleView($this->view($data, 201)
                ->setHeader('x-location', $this->generateUrl(
                'wi_api_v1_news_get_new', ['id' => $id])));
    }
}
