<?php

namespace Wi\Front\CoreBundle\Core\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\RouterInterface;
use WhiteOctober\BreadcrumbsBundle\Model\Breadcrumbs;
use Wi\Front\CoreBundle\Utils\Slugger;
use Symfony\Component\DependencyInjection\ContainerInterface;

class FrontController extends Controller
{
    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * @var Breadcrumbs
     */
    protected $breadcrumbs;

    /**
     * @var string
     */
    protected $defaultTemplate;

    /**
     * @var string
     */
    protected $template;

    /**
     * @var DelegatingEngine
     */
    protected $templating;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * Constructor
     *
     * @param Breadcrumbs $breadcrumbs
     * @param RouterInterface $router
     * @param Slugger $slugger
     */
    public function __construct(Breadcrumbs $breadcrumbs, RouterInterface $router, Slugger $slugger, ContainerInterface $container){
        $this->router = $router;
        $this->slugger = $slugger;
        $this->breadcrumbs = $breadcrumbs;
        $this->breadcrumbs->addItem("Home", $this->router->generate('homepage'));
        $this->defaultTemplate = $container->getParameter('default_template_name');
        $this->template = $container->getParameter('template_name');
        $this->templating = $container->get('templating');
    }

    /**
     * Slugify.
     *
     * @param   string $string
     * @return  string
     */
    protected function slugify($string)
    {
        return $this->slugger->slugify($string);
    }

    /**
     * Renderowanie odpowiedzi z cache'm.
     *
     * @param Response $response
     * @param string $type Typ elementu cache'owania
     * @return Response
     */
    protected function responseCache($response, $type)
    {
        $config = $this->cacheConfig($type);

        if (! empty($config) && boolval(intval($this->get('config')->get('cache.enable')))) {
            $response->setMaxAge($config['s-maxage']);
            $response->setSharedMaxAge($config['s-maxage']);

            if ($config['public']) {
                $response->setPublic();
            }
        }

        return $response;
    }

    /**
     * Zwraca plik szablonu. Jeśli nie istnieje plik to bierze szablon domyślny.
     *
     * @param string $template Ścieżka do szablonu.
     * @return string
     */
    protected function getTemplate(string $template) : string
    {
        $format = '@templates/%s/view/%s.html.twig';

        if (! $this->templating->exists($path = sprintf($format, $this->template, $template))) {
            return sprintf($format, $this->defaultTemplate, $template);
        }

        return $path;
    }

    // -------------------------------------------------------------------------

    /**
     * Pobranie konfiguracji dla cache.
     *
     * @param string $type Typ elementu cache'owania
     * @return array
     */
    private function cacheConfig($type)
    {
        $arr = [
            'max-age' => 0,
            's-maxage' => 0,
            'public' => false,
        ];
        $config = $this->get('config');

        switch ($type) {
            case 'news':
                $arr['max-age'] = (int) $config->get('cache.lifetimeNews');
                $arr['s-maxage'] = (int) $config->get('cache.lifetimeNews');
                $arr['public'] = true;
                break;
            case 'page':
                $arr['max-age'] = (int) $config->get('cache.lifetimePage');
                $arr['s-maxage'] = (int) $config->get('cache.lifetimePage');
                $arr['public'] = true;
                break;
        }

        return $arr;
    }
}
