<?php

namespace Wi\Front\NewsBundle\Controller;

use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Admin\NewsBundle\Entity\Category;
use Wi\Admin\NewsBundle\Entity\Comment;
use Wi\Admin\NewsBundle\Entity\CommentGuest;
use Wi\Admin\NewsBundle\Entity\News;
use Wi\Admin\NewsBundle\Service\Pdf;
use Wi\Front\CoreBundle\Core\Controller\FrontController;

/**
 * Blog controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class BlogController extends FrontController
{
    /**
     * List all news entities by category.
     *
     * @param   Request $request
     * @param   Category $category
     * @param   int $page
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request, Category $category = null, $page = 1)
    {
        if (! is_null($category)){
            $parents = array_reverse($this->getCategoryParents($category));
            foreach ($parents as $parent){
                $this->breadcrumbs->addItem($parent->getName(), $this->generateUrl("wi_front_blog_by_category_index", ['id' => $parent->getId(), 'slug' => $this->slugify($parent->getName())]));
            }
            $this->breadcrumbs->addItem($category->getName(), $this->generateUrl("wi_front_blog_by_category_index", ['id' => $category->getId(), 'slug' => $this->slugify($category->getName())]));
        } else {
            $this->breadcrumbs->addItem("Blog", $this->generateUrl("wi_front_blog_index"));
        }
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich aktywnych aktualności dla danej kategorii.
        $paginator = $this->get('knp_paginator');
        $news = $paginator->paginate(
            $em->getRepository('WiAdminNewsBundle:News')
                ->getFindAllActivePublishedQuery(News::TYPE_BLOG ,(! is_null($category) ? $category->getId() : null)),
            $request->query->getInt('page', $page),
            10
        );

        return $this->render($this->getTemplate('Blog/index'), [
            'category' => $category,
            'news' => $news,
        ]);
    }

    /**
     * Show a news entity.
     *
     * @param   Request $request
     * @param   News $news
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Request $request, News $news)
    {
        // Breadcrumbs.
        $this->breadcrumbs->addItem("Blog", $this->generateUrl("wi_front_blog_index"));
        $this->breadcrumbs->addItem($news->getTitle(), $this->generateUrl("wi_front_blog_show", ['id' => $news->getId(), 'slug' => $this->slugify($news->getTitle())]));

        // Increment visit counter, when user is not admin.
        if (! $this->isGranted('ROLE_ADMIN')) {
            $news->setVisitCounter($news->getVisitCounter() + 1);
            $this->getDoctrine()->getManager()->flush();
        }

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Ustawienie danych o gościu, jeśli jest informacja w ciasteczku.
        if ($cookie = $request->cookies->get('wi_guest')) {
            $guest = $em->getRepository('WiAdminNewsBundle:CommentGuest')->findOneByCookie($cookie);
        }

        // Pobranie komentarzy dla newsa.
        $comments = $em->getRepository('WiAdminNewsBundle:Comment')->findWithLimit($news);

        return $this->responseCache(
            $this->render($this->getTemplate('Blog/show'), [
                'comments' => $comments,
                'guest' => ($guest ?? null),
                'news' => $news,
            ]),
            'news'
        );
    }

    /**
     * Pobranie wszystkich komentarzy i zwrócenie wyrenderowanego szablonu.
     *
     * @param Request $request
     * @param News $news
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function loadCommentsAction(Request $request, News $news)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie komentarzy dla newsa.
        $comments = $em->getRepository('WiAdminNewsBundle:Comment')->findWithLimit($news);

        return $this->render('WiFrontNewsBundle:News:Components/comment_list.html.twig', [
            'comments' => $comments,
        ]);
    }

    /**
     * Dodawanie komentarzy.
     *
     * @param Request $request
     * @param News $news
     * @return JsonResponse
     */
    public function createCommentAction(Config $config, Request $request, News $news)
    {
        if (! $request->isXmlHttpRequest()) {
            return new JsonResponse([
                'status' => false,
                'message' => 'Not allowed',
            ]);
        }

        $commentPost = $request->request->get('Comment');
        $content = trim(strip_tags($commentPost['content']));
        $name = trim(strip_tags($commentPost['name'] ?? null));
        $email = trim(strip_tags($commentPost['email'] ?? null));
        $parent_id = intval(trim(strip_tags($commentPost['parent_id'])));

        if (empty($content)) {
            return new JsonResponse([
                'status' => false,
                'message' => 'Treść komentarza nie może być pusta',
            ]);
        }

        $limit = false;
        $limitValue = 0;
        $lastComment = null;
        $newCommentFor = 0;

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $comment = new Comment();

        // Ustawienie gościa, jeśli nie jest zalogowany użytkownik.
        if (! $this->isGranted('ROLE_USER')) {
            $cookie = md5(uniqid());
            $guest = $em->getRepository('WiAdminNewsBundle:CommentGuest')->findOneBy([
                'name' => $name,
                'email' => $email,
                'cookie' => $request->cookies->get('wi_guest'),
            ]);

            // Utworzenie nowego gościa, jeśli nie ma zapisanego go w bazie.
            if (! $guest) {
                $guest = new CommentGuest();
                $guest
                    ->setName($name)
                    ->setEmail($email)
                    ->setCookie($cookie)
                ;
            }
            $lastComment = $em->getRepository('WiAdminNewsBundle:Comment')->findOneBy(
                ['guest' => $guest],
                ['dateCreated' => 'DESC']
            );

            // Ustawienie limitu, jeśli istniał poprzedni komentarz użytkownika.
            if ($lastComment) {
                if (boolval(intval($config->get('comment.enableGuestLimit')))) {
                    $diff = time() - $lastComment->getDateCreated()->getTimestamp();
                    $limitValue = intval($config->get('comment.guestLimitValue'));

                    if ($diff <= $limitValue) {
                        $limit = true;
                        $newCommentFor = $limitValue - $diff;
                    }
                }
            }

            $comment->setGuest($guest);

            // Ustawienie statusu komentarza w zależności od polityki.
            if (! boolval(intval($config->get('comment.enableGuestAutoAccept')))) {
                $comment->setStatus(Comment::STATUS_WAIT);
            }
        }

        // Ustawienie użytkownika, jeśli jest zalogowany.
        if ($this->isGranted('ROLE_USER')) {
            $comment->setUser($this->getUser());
            $lastComment = $em->getRepository('WiAdminNewsBundle:Comment')->findOneBy(
                ['user' => $this->getUser()],
                ['dateCreated' => 'DESC']
            );

            // Ustawienie statusu komentarza w zależności od polityki.
            if (! boolval(intval($config->get('comment.enableUserAutoAccept')))) {
                $comment->setStatus(Comment::STATUS_WAIT);
            }

            // Ustawienie limitu, jeśli istniał poprzedni komentarz użytkownika.
            if ($lastComment) {
                if (boolval(intval($config->get('comment.enableUserLimit')))) {
                    $diff = time() - $lastComment->getDateCreated()->getTimestamp();
                    $limitValue = intval($config->get('comment.userLimitValue'));

                    if ($diff <= $limitValue) {
                        $limit = true;
                        $newCommentFor = $limitValue - $diff;
                    }
                }
            }
        }

        // Ustawienie rodzica komentarza.
        $parent = $em->getRepository('WiAdminNewsBundle:Comment')->find($parent_id);

        if (! is_null($parent)) {
            $parent = $parent->getParent() ?? $parent;
        }

        if ($limit) {
            $response = new JsonResponse([
                'status' => false,
                'message' => 'Nowy komentarz możesz za ' . $newCommentFor . 's',
            ]);
        } else {
            // Ustawienie szczegółów komentarza.
            $comment
                ->setNews($news)
                ->setParent($parent)
                ->setContent($content)
                ->setIp($request->server->get("REMOTE_ADDR"))
                ->setHost($request->server->get('REMOTE_HOST') ?? gethostbyaddr($request->server->get('REMOTE_ADDR')))
                ->setAgent($request->server->get('HTTP_USER_AGENT'))
            ;
            $em->persist($comment);
            $em->flush();

            $response = new JsonResponse([
                'status' => true,
                'message' => 'Komentarz został wysłany',
                'id' => $comment->getId(),
            ]);
        }


        if (! $this->isGranted('ROLE_USER')) {
            $response->headers->setCookie(new Cookie('wi_guest', $guest->getCookie(), strtotime('now + 1 year')));
        }

        return $response;
    }

    /**
     * Zapisanie aktualności/wpisu do PDF i pobranie.
     *
     * @param Pdf $pdf
     * @param Request $request
     * @param News $news
     * @return BinaryFileResponse
     */
    public function pdfAction(Pdf $pdf, Request $request, News $news)
    {
        return $this->file($pdf->generate($pdf->setNews($news)->download()->html()));
    }

    // -------------------------------------------------------------------------

    /**
     * Get parent of Category
     *
     * @param Category $category
     * @return table $parents
     */
    private function getCategoryParents(Category $category, $parents = []){
        $parent = $category->getParent();
        if(! is_null($parent)){
            $parents[] = $parent;
            $parents = $this->getCategoryParents($parent, $parents);
        }
        return $parents;
    }

}
