<?php

namespace Wi\Front\UserBundle\Controller;

use Wi\Front\CoreBundle\Core\Controller\FrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\UserBundle\Entity\BillingData;
use Wi\Admin\UserBundle\Entity\Profile;
use Wi\Admin\UserBundle\Entity\User;
use Wi\Front\UserBundle\Form\BillingDataType;
use Wi\Front\UserBundle\Form\UserPasswordType;
use Wi\Front\UserBundle\Form\ProfileType;
use Wi\Admin\UserBundle\User\FavoriteManager;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Wi\Admin\UserBundle\Entity\UserAvatar;
use Wi\Admin\UserBundle\Entity\UserInterest;
use Wi\Front\UserBundle\Form\Model\ChangePassword;
use Wi\Front\UserBundle\Form\UserAvatarType;
use Wi\Front\UserBundle\Form\UserEmailType;
use Wi\Front\UserBundle\Form\UserInterestType;

/**
 * User Controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class UserController extends FrontController
{
    /**
     * Edycja danych użytkownika.
     *
     * @param   Request $request
     * @param   User $user
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $forms = $this->handleForms($request, $em);
        $user = $this->getUser();

        foreach ($forms as $form) {
            if ($form instanceof RedirectResponse) {
                return $form;
            }
        }


        return $this->render($this->getTemplate('Users/User/edit'), array_merge($forms, [
            'user' => $user,
        ]));
    }

    /**
     * Dodanie do ulubionych. Metoda GET.
     *
     * @param Request $request
     * @param FavoriteManager $favoriteManager
     * @return JsonResponse
     */
    public function ajaxFavoriteAddAction(Request $request, FavoriteManager $favoriteManager)
    {
        $user_id = $this->getUser()->getId();
        $type_id = $request->query->getInt('type_id');
        $object_id = $request->query->getInt('object_id');

        $result = $favoriteManager->add($type_id, $user_id, $object_id);

        return new JsonResponse([
            'status' => (int) $result,
        ]);
    }

    /**
     * Usunięcie z ulubionych. Metoda GET.
     *
     * @param Request $request
     * @param FavoriteManager $favoriteManager
     * @return JsonResponse
     */
    public function ajaxFavoriteRemoveAction(Request $request, FavoriteManager $favoriteManager)
    {
        $user_id = $this->getUser()->getId();
        $type_id = $request->query->getInt('type_id');
        $object_id = $request->query->getInt('object_id');

        $result = $favoriteManager->remove($type_id, $user_id, $object_id);

        return new JsonResponse([
            'status' => (int) $result,
        ]);
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Przetwarzanie formularzy.
     *
     * @param Request $request
     * @param EntityManager $em
     * @return array
     */
    private function handleForms(Request $request, $em)
    {
        $user = $this->getUser();

        return [
            'billingDataForm' => $this->handleFormBillingData($request, $em, $user),
            'emailForm' => $this->handleFormEmail($request, $em, $user),
            'passwordForm' => $this->handleFormPassword($request, $em, $user),
            'profileForm' => $this->handleFormProfile($request, $em, $user),
            'userAvatarsForm' => $this->handleFormAvatar($request, $em, $user),
            'userInterestsForm' => $this->handleFormInterest($request, $em, $user),
        ];
    }

    /**
     * Obsługa formularza edycji avatara.
     *
     * @param Request $request
     * @param EntityManager $em
     * @param User $user
     * @return FormView|RedirectResponse
     */
    private function handleFormAvatar(Request $request, $em, $user)
    {
        // Sprawdzenie, czy istnieje obiekt avataru użytkownika.
        if (! $avatar = $user->getUserAvatars()->first()) {
            // Utworzenie nowego obiektu danych profilowych.
            $avatar = new UserAvatar();

            // Przypisanie użytkownika.
            $avatar->setUser($user);
        }

        // Generowanie formularza edycji avataru użytkownika
        $userAvatarsForm = $this->createForm(UserAvatarType::class, $user);
        $userAvatarsForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($userAvatarsForm->isSubmitted()) {
            if ($userAvatarsForm->isValid()) {
                // Pobranie danych z formularza.
                $avatar = $userAvatarsForm->get('avatar')->getData();

                if ($lastAvatar = $user->getUserAvatars()->first()) {
                    // Usunięcie starego avatara.
                    $em->remove($lastAvatar);
                }

                // Utworzenie nowego avatara.
                $newAvatar = new UserAvatar();
                $newAvatar
                    ->setUser($user)
                    ->setAvatar($avatar)
                ;

                // Przygotowanie do zapisu.
                $em->persist($newAvatar);

                // Upload file.
                if ($userAvatarsForm->get('image')->getData()) {
                    $userUpload = $this->get('wi.admin.user.upload');
                    $filename = $userUpload->upload(
                        $userAvatarsForm->get('image')->getData(),
                        [
                            'cropForce' => true,
                        ]
                    );

                    // Set filename.
                    $user->setImage($filename);

                    // Przygotowanie do zapisu.
                    $em->persist($user);
                }

                // Zapis.
                $em->flush();

                // Zapisanie logu o zmianie avatara.
                $this->get('wi.front.user.logger')->create(14, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'Avatar has been changed.',
                ]);

                return $this->redirectToRoute('wi_front_user_edit');
            } else {
                // Informacja o błędnej walidacji formularza
                $this->addFlash('validate.error', 'avatar');

                // Zapisanie logu o błędzie podczas zmiany avatara.
                $this->get('wi.front.user.logger')->create(15, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'An error occurred while changing the interests.',
                ]);
            }
        }

        return $userAvatarsForm->createView();
    }

    /**
     * Obsługa formularza edycji danych bilingowych.
     *
     * @param Request $request
     * @param EntityManager $em
     * @param User $user
     * @return FormView|RedirectResponse
     */
    private function handleFormBillingData(Request $request, $em, $user)
    {
        // Sprawdzenie, czy istnieje obiekt danych billingowych użytkownika.
        if (! $billingData = $user->getBillingData()->first()) {
            // Utworzenie nowego obiektu danych billingowych.
            $billingData = new BillingData();

            // Przypisanie użytkownika.
            $billingData->setUser($user);
        }

        // Generowanie formularza edycji danych billingowych.
        $billingDataForm = $this->createForm(BillingDataType::class, $billingData);
        $billingDataForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($billingDataForm->isSubmitted()) {
            if ($billingDataForm->isValid()) {
                // Remove white space from strings.
                $billingData->setZip(preg_replace('/\s/', '', $billingData->getZip()));
                $billingData->setNIP(preg_replace('/\s/', '', $billingData->getNIP()));

                if ($lastBillingData = $user->getBillingData()->first()) {
                    // Usunięcie starych danych billingowych.
                    $em->remove($lastBillingData);

                    // Utworzenie nowych danych billingowych.
                    $newBillingData = new BillingData();
                    $newBillingData
                        ->setName($billingData->getName())
                        ->setAddress($billingData->getAddress())
                        ->setZip($billingData->getZip())
                        ->setCity($billingData->getCity())
                        ->setCountry($billingData->getCountry())
                        ->setNIP($billingData->getNIP())
                        ->setUser($user)
                    ;
                    $billingData = $newBillingData;

                }

                // Dodanie prolifu do użytkownika.
                $user->addBillingData($billingData);

                // Zapis.
                $em->persist($billingData);
                $em->flush();

                // Zapisanie logu o zmianie danych profilowych.
                $this->get('wi.front.user.logger')->create(8, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'Billing data has been changed.',
                ]);

                return $this->redirectToRoute('wi_front_user_edit');
            } else {
                // Informacja o błędnej walidacji formularza
                $this->addFlash('validate.error', 'billing');

                // Zapisanie logu o błędzie podczas zmiany danych profilowych.
                $this->get('wi.front.user.logger')->create(9, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'An error occurred while changing the billing data.',
                ]);
            }
        }

        return $billingDataForm->createView();
    }

    /**
     * Obsługa formularza edycji maila.
     *
     * @param Request $request
     * @param EntityManager $em
     * @param User $user
     * @return FormView|RedirectResponse
     */
    private function handleFormEmail(Request $request, $em, $user)
    {
        $emailForm = $this->createForm(UserEmailType::class, $user);
        $emailForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($emailForm->isSubmitted()) {
            if ($emailForm->isValid()) {
                $user->setHashCode(null);

                // Zapis.
                $em->persist($user);
                $em->flush();

                return $this->redirectToRoute('wi_front_user_edit');
            } else {
                // Informacja o błędnej walidacji formularza
                $this->addFlash('validate.error', 'email');
            }
        }

        return $emailForm->createView();
    }

    /**
     * Obsługa formularza edycji zainteresowań.
     *
     * @param Request $request
     * @param EntityManager $em
     * @param User $user
     * @return FormView|RedirectResponse
     */
    private function handleFormInterest(Request $request, $em, $user)
    {
        // Generowanie formularza edycji zainteresowań użytkownika
        $userInterestsForm = $this->createForm(UserInterestType::class, $user);
        $userInterestsForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($userInterestsForm->isSubmitted()) {
            if ($userInterestsForm->isValid()) {
                $interests = $userInterestsForm->get('interests')->getData();

                // Sprawdzenie, czy przesłane opcje nie są puste.
                if  (! empty($interests)) {
                    foreach ($interests as $interest) {
                        // Sprawdzenie, czy użytkownik posiada takie zainteresowanie.
                        $hasInterest = $user->getUserInterests()->filter(
                            function ($entry) use ($interest) {
                                return $entry->getInterest() === $interest;
                            }
                        );

                        // Sprawdzenie, czy użytkownik nie posiada takiego zainteresowania.
                        if ($hasInterest->count() == 0) {
                            // Utworzenie nowego obiektu.
                            $userInterest = new UserInterest();

                            // Przypisanie danych.
                            $userInterest
                                ->setUser($user)
                                ->setInterest($interest)
                            ;

                            $em->persist($userInterest);
                        }
                    }
                }

                foreach ($user->getUserInterests() as $userInterest) {
                    $hasUserInterest = $interests->contains($userInterest->getInterest());

                    // Sprawdzenie, czy usunięto zainteresowania.
                    if (! $hasUserInterest) {
                        $em->remove($userInterest);
                    }
                }

                // Przygotowanie do zapisu i zapis.
                $em->persist($user);
                $em->flush();

                // Zapisanie logu o zmianie zainteresowań.
                $this->get('wi.front.user.logger')->create(12, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'Interests has been changed.',
                ]);

                return $this->redirectToRoute('wi_front_user_edit');
            } else {
                // Informacja o błędnej walidacji formularza
                $this->addFlash('validate.error', 'interest');

                // Zapisanie logu o błędzie podczas zmiany zainteresowań.
                $this->get('wi.front.user.logger')->create(13, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'An error occurred while changing the interests.',
                ]);
            }
        }

        return $userInterestsForm->createView();
    }

    /**
     * Obsługa formularza edycji hasła.
     *
     * @param Request $request
     * @param EntityManager $em
     * @param User $user
     * @return FormView|RedirectResponse
     */
    private function handleFormPassword(Request $request, $em, $user)
    {
        $changePassword = new ChangePassword();

        // Generowanie formularza edycji zmiany hasła użytkownika.
        $passwordForm = $this->createForm(UserPasswordType::class, $changePassword);
        $passwordForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($passwordForm->isSubmitted()) {
            if ($passwordForm->isValid()) {
                // Ustawienie hasła.
                $user->setPassword(
                    $this->get('security.password_encoder')
                        ->encodePassword(
                            $user,
                            $changePassword->getNewPassword()
                        )
                );

                // Zapis.
                $em->persist($user);
                $em->flush();

                // Zapisanie logu o zmianie hasła.
                $this->get('wi.front.user.logger')->create(6, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'Password has been changed.',
                ]);

                return $this->redirectToRoute('wi_front_user_edit');
            } else {
                // Informacja o błędnej walidacji formularza
                $this->addFlash('validate.error', 'password');

                // Zapisanie logu o błędzie podczas zmiany hasła.
                $this->get('wi.front.user.logger')->create(7, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'An error occurred while changing the password.',
                ]);
            }
        }

        return $passwordForm->createView();
    }

    /**
     * Obsługa formularza edycji danych profilowych.
     *
     * @param Request $request
     * @param EntityManager $em
     * @param User $user
     * @return FormView|RedirectResponse
     */
    private function handleFormProfile(Request $request, $em, $user)
    {
        // Sprawdzenie, czy istnieje obiekt danych profilowych użytkownika.
        if (! $profile = $user->getProfile()->first()) {
            // Utworzenie nowego obiektu danych profilowych.
            $profile = new Profile();

            // Przypisanie użytkownika.
            $profile->setUser($user);
        }

        // Genreowanie formularza edycji danych profilowych.
        $profileForm = $this->createForm(ProfileType::class, $profile);
        $profileForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($profileForm->isSubmitted()) {
            if ($profileForm->isValid()) {
                if ($lastProfile = $user->getProfile()->first()) {
                    // Usunięcie starych danych profilowych.
                    $em->remove($lastProfile);

                    // Utworzenie nowych danych profilowych.
                    $newProfile = new Profile();
                    $newProfile
                        ->setFirstName($profile->getFirstName())
                        ->setLastName($profile->getLastName())
                        ->setSex($profile->getSex())
                        ->setDateOfBirth($profile->getDateOfBirth())
                        ->setDescription($profile->getDescription())
                        ->setPesel($profile->getPesel())
                        ->setUser($user)
                    ;
                    $profile = $newProfile;
                }

                // Dodanie profilu do użytkownika.
                $user->addProfile($profile);

                // Zapis.
                $em->persist($profile);
                $em->flush();

                // Zapisanie logu o zmianie danych profilowych.
                $this->get('wi.front.user.logger')->create(10, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'Profile data has been changed.',
                ]);

                return $this->redirectToRoute('wi_front_user_edit');
            } else {
                // Informacja o błędnej walidacji formularza
                $this->addFlash('validate.error', 'profile');

                // Zapisanie logu o błędzie podczas zmiany danych profilowych.
                $this->get('wi.front.user.logger')->create(11, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'An error occurred while changing the profile data.',
                ]);
            }
        }

        return $profileForm->createView();
    }

    /**
     * Usunięcie zdjecia użytkownika.
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function removeImageAction()
    {
        $user = $this->getUser();
        $user->setImage(null);

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $em->persist($user);
        $em->flush();

        return $this->redirectToRoute('wi_front_user_edit');
    }
}
